# ------------------------------------------------------------------------
# 
# "jackVarBlock(x, func, length.block, ...)" --
# 
# Jackknife estimator by blocks for the variance of function 'func'.
# 
# ------------------------------------------------------------------------
##
#' @aliases jackVarBlock
#' @title Jackknife Variance Estimator Based on Fixed Length Blocks
#' @description 
#' Estimates the variance of a statistic applied to a vector or a matrix 
#' using a block jackknife procedure.
#' @param x A vector or a matrix representing the data.
#' @param func The function used to compute the statistic on each sample.
#' @param length.block An integer; the block length. 
#' @param ... Optional additional arguments for the \code{func} function.
#' @details 
#' The \code{jackVarBlock} function extends the jackknife variance estimation to block data. 
#' The data \code{x} is divided into non-overlapping blocks of length \code{length.block}. 
#' The function \code{func}, having output size equal to \emph{p}, 
#' is applied to the data with each block removed in turn to finally compute an empirical
#' variance of the obtained values. This approach is particularly useful 
#' for dependent data, where traditional jackknife methods may not be appropriate.
#'
#' 
#' @return 
#' Returns a scalar or a covariance matrix, depending on whether the function \code{func} 
#' is univariate or multivariate. For a function returning a vector of length 
#' \emph{p}, the output will be a covariance matrix of size \emph{p x p}.
#' 
#' @references 
#' Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Carlstein, E. (1986). The use of subseries methods for estimating the variance 
#' of a general statistic from a stationary time series. \emph{Annals of Statist.},
#' \bold{14}, 1171-1179.
#' 
#' Gray, H., Schucany, W. and Watkins, T. (1972). 
#' \emph{The Generalized Jackknife Statistics}. 
#' Marcel Dekker, New York.
#' 
#' Quenouille, M.H. (1949). Approximate tests of correlation in time-series. 
#' \emph{J. Roy. Statist. Soc., Ser. B}, \bold{11}, 68-84.
#' 
#' 
#' @seealso 
#' \code{\link{jackVar}}, 
#' \code{\link{jackFunc}},
#' \code{\link{blockboot}},
#' \code{\link{jackFuncBlock}}, 
#' \code{\link{jackFuncRegen}}.
#' 
#' @keywords 
#' Jackknife Resampling "Statistical analysis" Blocks
#' 
#' @export
#' @examples 
#' set.seed(1)
#' x <- arima.sim(list(order = c(1, 0, 4), ar = 0.5, ma = c(0.7, 0.4, -0.3, -0.1)), n = 101)
#' # Jackknife variance estimator of 'func' with blocks of length length.block
#' length.block <- 10
#' V1 <- jackVarBlock(x, mean, length.block)
##

jackVarBlock <- function(x,func,length.block,...) {
  # Test the value returned by func
  
  Tn <- func(x,...)
  if (!is.vector(Tn)) {
    stop("Function 'func' must return a vector")
  }
  n <- length(x)
  if (n <= length.block) {stop("length.block must be less than length of x")}	
  bnum <- n%/%length.block
  p <- length(Tn)
  Tnames <- names(Tn)	
  J <- matrix(nrow=bnum,ncol=p)
  for (i in 1:bnum) {
    # Indices of the block to remove
    bidx <- ((i-1)*length.block+1):(i*length.block)
    Tni <- func(x[-bidx],...)
    J[i,] <- (n*Tn - (n-length.block)*Tni)/length.block
  }
  muJ <- colMeans(J)
  V <- matrix(0,nrow=p,ncol=p)
  for (i in 1:bnum) {
    L <- J[i,]-muJ
    V <- V + L%*%t(L)
  }
  V <- length.block*V/(n*bnum)
  if (!is.null(Tnames)) {
    rownames(V) <- Tnames
    colnames(V) <- Tnames
  }
  if (p == 1) {V <- as.vector(V)}
  return(V)
}
