% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bioregion_colors.R
\name{bioregion_colors}
\alias{bioregion_colors}
\title{Add color palettes to bioregion cluster objects}
\usage{
bioregion_colors(
  clusters,
  palette = "Vivid",
  cluster_ordering = "n_sites",
  cutoff_insignificant = NULL
)
}
\arguments{
\item{clusters}{An object of class \code{bioregion.clusters}, typically output
from clustering functions such as \code{\link[=netclu_infomap]{netclu_infomap()}}, \code{\link[=hclu_hierarclust]{hclu_hierarclust()}},
or \code{\link[=nhclu_pam]{nhclu_pam()}}.}

\item{palette}{A \code{character} string indicating which color palette from
\code{rcartocolor} to use. Default is \code{"Vivid"}. Other qualitative palettes
include \code{"Bold"}, \code{"Prism"}, \code{"Safe"}, \code{"Antique"}, and \code{"Pastel"}.}

\item{cluster_ordering}{A \code{character} string indicating the criterion for
ranking clusters to determine color assignment priority. Options are:
\itemize{
\item \code{"n_sites"} (default): Rank by number of sites in each cluster
\item \code{"n_species"}: Rank by number of species (bipartite networks only)
\item \code{"n_both"}: Rank by combined sites + species (bipartite networks only)
}
Larger clusters (by the chosen criterion) receive vivid colors first.}

\item{cutoff_insignificant}{A \code{numeric} value or \code{NULL} (default). When
specified, clusters with values at or below this threshold (based on the
\code{cluster_ordering} criterion) are considered insignificant and colored
black, reducing visual clutter on maps. If \code{NULL}, all clusters receive
distinct colors.}
}
\value{
A modified \code{bioregion.clusters} object with two additional elements:
\itemize{
\item \code{colors}: A \code{list} where each element corresponds to a partition
(bioregionalization). Each list element is a \code{data.frame} with two columns:
\itemize{
\item \code{cluster} (\code{character}): Cluster identifier for that partition
\item \code{color} (\code{character}): Hex color code (e.g., "#FF5733")
}
\item \code{clusters_colors}: A \code{data.frame} with the same structure as the
\code{clusters} element, but with cluster IDs replaced by their corresponding
hex color codes for direct use in plotting functions.
}
}
\description{
This function assigns colors to clusters in a \code{bioregion.clusters} object
using color palettes from the \code{rcartocolor} package. It
handles large numbers of clusters by assigning vivid colors to the most
important clusters (based on size), grey shades to less important clusters,
and optionally black to insignificant clusters.
}
\details{
The function uses a two-step algorithm to assign colors:

\strong{Step 1: Identify insignificant clusters} (if \code{cutoff_insignificant} is specified)

Insignificant clusters are those with a marginal size compared to others.
This is a subjective threshold set by the user. All such clusters are assigned
the color black (#000000) to minimize their visual impact.
Clusters with values at or below the threshold are assigned black (#000000).

\strong{Step 2: Assign colors to significant clusters}

Remaining clusters are ranked by the \code{cluster_ordering} criterion:
\itemize{
\item \strong{Top clusters} (up to 12): Receive distinct colors from the chosen
palette. This limit is because above 12 the human eye struggles to
distinguish between colors.
\item \strong{Remaining clusters} (beyond top 12): Receive shades of grey from
light (#CCCCCC) to dark (#404040), maintaining visual distinction but
with less prominence.
}

\strong{Multiple partitions}: If the cluster object contains multiple partitions
(e.g., from hierarchical clustering with different k values), colors are
assigned independently for each partition. Each partition gets its own color
scale optimized for the number of clusters in that partition.
}
\note{
The colored cluster object can be directly used with \code{\link[=map_bioregions]{map_bioregions()}},
which will automatically detect and apply the color scheme when present.
}
\examples{
data(fishmat)
data(fishsf)

# Basic example with few clusters
sim <- similarity(fishmat, metric = "Simpson")
clust <- netclu_greedy(sim)
clust_colored <- bioregion_colors(clust)
print(clust_colored)

\dontrun{
# Map with automatic colors
map_bioregions(clust_colored, fishsf)

# Example with many clusters and cutoff
dissim <- similarity_to_dissimilarity(sim)
clust <- hclu_hierarclust(dissim,
                          optimal_tree_method = "best",
                          n_clust = 15)
clust_colored2 <- bioregion_colors(clust, 
                                   cluster_ordering = "n_sites",
                                   cutoff_insignificant = 1)
map_bioregions(clust_colored2, fishsf)

# Example with different palette
clust_colored3 <- bioregion_colors(clust, palette = "Bold")
map_bioregions(clust_colored3, fishsf)


# Example with bipartite network
clust_bip <- netclu_greedy(fishdf, bipartite = TRUE)
clust_bip_colored <- bioregion_colors(clust_bip, 
                                      cluster_ordering = "n_both")
map_bioregions(clust_bip_colored, fishsf)
                                      
}

}
\references{
Color palettes from the \code{rcartocolor} package:
Nowosad J (2018). "CARTOColors: color palettes inspired by CARTO."
\url{https://github.com/Nowosad/rcartocolor}
}
\seealso{
\code{\link[=map_bioregions]{map_bioregions()}} for visualizing colored clusters on maps
}
\author{
Boris Leroy (\email{leroy.boris@gmail.com}) \cr
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com})
}
