test_that("input_validation", {
  ##############################################################################
  # Tests checking input validation of epsilon
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = -1),
    "If sample_G is a function, epsilon must be specified as a single numeric value strictly between 0 and 1.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 1.2),
    "If sample_G is a function, epsilon must be specified as a single numeric value strictly between 0 and 1.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = "String"),
    "If sample_G is a function, epsilon must be specified as a single numeric value strictly between 0 and 1.", fixed=TRUE
  )

  ##############################################################################
  # Tests checking input validation of max_samples
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, max_samples = -2),
    "max_samples must be a single integer greater or equal to 1.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, max_samples = 1.2),
    "max_samples must be a single integer greater or equal to 1.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, max_samples =  "string"),
    "max_samples must be a single integer greater or equal to 1.", fixed=TRUE
  )

  ##############################################################################
  # Tests checking input validation of min_samples
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, min_samples = -1),
    "min_samples must be a single integer greater or equal to 0 and smaller than max_samples.", fixed = TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, min_samples = 1.5),
    "min_samples must be a single integer greater or equal to 0 and smaller than max_samples.", fixed = TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, min_samples = "string"),
    "min_samples must be a single integer greater or equal to 0 and smaller than max_samples.", fixed = TRUE
  )

  ##############################################################################
  # Tests checking input validation of stopcrit
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, stopcrit = 3),
    "Invalid stopping criterion specified.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001, stopcrit = function(){FALSE}),
    "Custom stopping criterion must be specified through a function taking a single argument with class avseqmc_progress.", fixed=TRUE
  )

  ##############################################################################
  # Tests checking input validation of sample_G
  expect_error(
    avseqmc(sample_G = function(a){runif(1) < a}, epsilon = 0.001),
    "If sample_G is a function, it must not take any arguments.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = 4, epsilon = 0.001),
    "sample_G must be either a function with no arguments, or an object of class 'avseqmc_progress'.", fixed=TRUE
  )
  expect_error(
    avseqmc(sample_G = list("ptilde"=0.03), epsilon = 0.001),
    "sample_G must be either a function with no arguments, or an object of class 'avseqmc_progress'.", fixed=TRUE
  )
  expect_warning(
    avseqmc(sample_G = init_avseqmc_progress(function(){runif(1) < 0.01}, 0.001), epsilon = 0.001),
    "epsilon is ignored when sample_G is of class 'avseqmc_progress'.", fixed=TRUE
  )
})



test_that("stopping_criterion_validity_futility", {

  # Test checking that futility criterion was reached at the correct time
  set.seed(1234)
  R1 <- avseqmc(sample_G = function(){runif(1) < 0.01}, epsilon = 0.001)
  expect_equal(R1$ptilde[length(R1$ptilde)] < 0.05 & R1$ptilde[length(R1$ptilde)-1] > 0.05, TRUE)
  # Test it does not continue if prompted with same convergence criterion
  R2 = avseqmc(R1)
  expect_equal(R2$n, R1$n)

  # Same but now with batch sampling
  R3 <- avseqmc(sample_G = function(){runif(10) < 0.01}, epsilon = 0.001)
  expect_equal(R3$ptilde[length(R3$ptilde)] < 0.05 & R3$ptilde[length(R3$ptilde)-1] > 0.05, TRUE)
  # Test it does not continue if prompted with same convergence criterion
  R4 = avseqmc(R3)
  expect_equal(R4$n, R3$n)

})


test_that("stopping_criterion_validity_convergence", {

  # Test checking that futility criterion was reached at the correct time
  set.seed(1234)
  gamma = 1e-5
  n0 = 100
  R1 <- avseqmc(sample_G = function(){runif(1) < 0.01}, stopcrit = list("type"="convergence", param=c(gamma, n0)), epsilon = 0.001)
  crit = (R1$ptilde[length(R1$Bn) - which(cumsum(rev(R1$Bn)) >= n0)[1] + 1] - R1$ptilde[length(R1$ptilde)]) / n0 < gamma
  expect_equal(crit,TRUE)
  # Test it does not continue if prompted with same convergence criterion
  R2 = avseqmc(R1)
  expect_equal(R2$n, R1$n)

  # Same but now with batch sampling
  gamma = 1e-5
  n0 = 100
  R3 <- avseqmc(sample_G = function(){runif(15) < 0.01}, stopcrit = list("type"="convergence", param=c(gamma, n0)), epsilon = 0.001)
  crit = (R3$ptilde[length(R3$Bn) - which(cumsum(rev(R3$Bn)) > n0)[1] + 1] - R3$ptilde[length(R3$ptilde)]) / n0 < gamma
  expect_equal(crit,TRUE)
  # Test it does not continue if prompted with same convergence criterion
  R4 = avseqmc(R3, stopcrit = list("type"="convergence", param=c(gamma, n0)))
  expect_equal(R4$n, R3$n)
})


test_that("stopping_criterion_mixing", {

  # Test that, although in the second part of the sampling we will not compute lower bounds,
  # the vectors of ptilde and the lower bound remain of the same size (although lower
  # bound will have some NA values when it is not computed)

  sample_G = function(){return(runif(1) < 0.01)}
  gamma = 1e-5
  n0 = 100
  set.seed(123)
  R1 = avseqmc(sample_G, 0.01)
  R2 = avseqmc(R1, stopcrit = list("type"="convergence","param"=c(gamma,n0)), max_samples=1000)
  R3 = avseqmc(R2, stopcrit = list("type"="futility","param"=0.01), max_samples=1000)

  expect_equal(length(R1$ptilde), length(R1$Ltilde))
  expect_equal(length(R2$ptilde), length(R2$Ltilde))
  expect_equal(length(R3$ptilde), length(R3$Ltilde))

})



test_that("stopping_criterion_min_max_samples_Works", {

  # Test that when both min and max is specified, test runs until max if stopcrit
  # cannot be reached prior
  set.seed(123)
  sample_G1 = function(){return(runif(1) < 0.05)}
  R1 = avseqmc(sample_G1, 0.01, min_samples= 50, max_samples = 100)
  expect_equal(R1$n, 100)

  # Test that when only min is specified test runs until min even if stopcrit is
  # reached before time
  sample_G2 = function(){return(0)}
  R2 = avseqmc(sample_G2, 0.01, min_samples= 1010)
  expect_equal(R2$n,1010)

  # Test that when only max is specified test does not run until max but rather
  # when stopcrit is reached
  R3 = avseqmc(sample_G2, 0.01, min_samples= 50, max_samples = 1000)
  expect_equal(R3$n, 248)


})

