#'@title Initialize an `avseqmc_progress` Object
#'
#'@description Initialize an object of class `avseqmc_progress`. This object
#'  contains the progress of an earlier sequentially anytime-valid estimated
#'  p-value from a Monte-Carlo simulation as described in Stoepker and Castro
#'  (2024, Definition 6). Subsequent references to equations and sections in
#'  this section of the reference manual pertain to this paper.
#'
#'  The object can be passed to `avseqmc()` to resume sampling.
#'
#'@param sample_G A function (without arguments) that draws one (or a batch of)
#'  zero/one samples from the distribution G as in Equation (5), where the
#'  function returns a vector of zeroes and ones.
#'@param epsilon The desired risk of overestimated significance.
#'@param ptilde Optional: a previously computed anytime-valid p-value estimate
#'  (via the construction in Definition 6, based on Robbins (1970)). If
#'  supplied, `n` must also be provided. The number of previously observed ones
#'  observed from the Monte-Carlo sampler `sample_G` (`S`) is inferred from
#'  `ptilde` and `n` unless given explicitly. If `ptilde` is rounded, supplying
#'  `S` is recommended to avoid numerical inaccuracies in inferring it.
#'@param n Optional: The number of Monte-Carlo samples used to compute the
#'  supplied `ptilde`. Must be given if `ptilde` or `S` is provided.
#'@param S Optional: The number of ones observed from `sample_G`. Must be given
#'  when `n` is supplied. Can be used instead of inferring it from `ptilde` and
#'  `n`.
#'
#'@return An object of class `avseqmc_progress` containing the progress of the
#'  sequentially estimated p-value. The object is a list containing the
#'  following elements:
#'
#'  - `$epsilon`: risk of overestimated significance used in the sequential estimation.
#'  - `$sample_G`: function that samples (batches) from the Monte-Carlo distribution $G^*(X)$ as in Equation (5).
#'  - `$ptilde`: sequence of sequential $p$-value estimates. The final value in this sequence is the most recent estimate of the $p$-value.
#'  - `$Ltilde`: sequence of lower bounds of the confidence sequence based on the construction by Robbins (1970). Contains NA values if these were
#'  not computed by default through `stopcrit =
#'  list("type"="futility","param"=...)` or requested using
#'  `compute_lower=TRUE`.
#'  - `$n`: total number of samples drawn from the MC sampler.
#'  - `$S`: total number of ones observed from the MC sampler.
#'  - `$B`: sequence of number of ones observed at each sampling timepoint (which can be greater than 1 if `sample_G` samples in batches)
#'  - `$Bn`: sequence of number of samples drawn from MC sampler at each timepoint (which can be greater than 1 if `sample_G` samples in batches)
#'
#'  If `ptilde`, `n`, and `S` are all empty, the object is initialized but empty
#'  and can be used as a starting point for sequential estimation.
#'
#' @examples
#' # Minimal example to construct an object based on earlier values of n and S
#' G1 <- function(){runif(1) < 0.04}
#' R1 <- init_avseqmc_progress(sample_G = G1,
#'                             epsilon = 0.001,
#'                             n = 1000,
#'                             S = 44)
#'
#'@seealso \code{\link{avseqmc}} which can be used to resume the anytime-valid
#'  sampling as logged in the constructed object.
#'
#'@references Stoepker, I. V., and R. M. Castro. 2024. Inference with Sequential
#'  Monte-Carlo Computation of p-Values: Fast and Valid Approaches.
#'  https://doi.org/10.48550/arXiv.2409.18908.
#'
#'  Robbins, H. (1970). Statistical Methods Related to the Law of the Iterated
#'  Logarithm. The Annals of Mathematical Statistics, 41(5):1397–1409.
#'  http://dx.doi.org/10.1214/aoms/1177696786
#'
#'@export
init_avseqmc_progress <- function(sample_G, epsilon, ptilde = NULL, n = 0, S = 0) {

  ##############################################################################
  # Input validation: argument types
  if (!missing(sample_G)) {
    if (!is.function(sample_G)) {
      stop("Invalid input: sample_G must be a function.")
    }
    if (length(formals(sample_G)) != 0) {
      stop("Invalid input: sample_G must be a function that takes no arguments.")
    }
  }
  if (!missing(epsilon)) {
    if (!is.numeric(epsilon) || length(epsilon) != 1 || epsilon <= 0 || epsilon >= 1) {
      stop("Invalid input: epsilon must be a single number strictly between 0 and 1.")
    }
  }
  if (!missing(ptilde)) {
    if (!is.numeric(ptilde) || length(ptilde) != 1 || ptilde < 0 || ptilde > 1) {
      stop("Invalid input: ptilde must be a single number between 0 and 1.")
    }
  }
  if (!missing(n)) {
    if (!is.numeric(n) || length(n) != 1 || n %% 1 != 0 || n < 1) {
      stop("Invalid input: n must be a single integer greater than or equal to 1.")
    }
  }
  if (!missing(S)) {
    if (!is.numeric(S) || length(S) != 1 || S %% 1 != 0 || S < 0) {
      stop("Invalid input: S must be a single integer greater than or equal to 1.")
    }
  }

  ##############################################################################
  # Input validation: combinations of arguments correct.
  # Check if sample_G and epsilon are not missing
  if(missing(sample_G) || missing(epsilon)){
    stop("To initialize a avseqmc_progress object, at least a MC sampling function sample_G and risk of overestimated significance epsilon must be specified.")
  }

  empty <- FALSE
  # Check which variables are present
  if (missing(n) && missing(S) && missing(ptilde)) {
    # No history is provided, so an empty object will be created.
    empty <- TRUE

  } else if (!missing(n) && missing(S) && !missing(ptilde)) {
    # n + ptilde present
    warning("Initializing avseqmc_progress with a previous value of ptilde, without the previously observed number of ones S from the MC function sample_G. The value of S is computed from ptilde, but may be inaccurate if ptilde is a rounded value.")
    # Compute S from ptilde
    S <- invert_Un_tilde(ptilde-epsilon,n,epsilon)

  } else if (!missing(n) && !missing(S) && missing(ptilde)) {
    # n + S present. Compute ptilde.
    ptilde <- Un_tilde(S, n, epsilon) + epsilon

  } else if (!missing(n) && !missing(S) && !missing(ptilde)) {
    # n + S + ptilde present
    warning("Initializing avseqmc_progress with a previous value of n, S and ptilde. Note that ptilde is not recomputed and is assumed correct.")

  } else {
    # Any other combination
    stop("Invalid input. To initialize a avseqmc_progress object with earlier simulation progress, n must be provided with ptilde and/or S.")
  }

  if(empty){
    B <- NULL
    Bn <- NULL
    Ltilde <- NULL
  } else {
    B <- S
    Bn <- n
    Ltilde <- Ln_tilde(S, n, epsilon)
  }

  ##############################################################################
  # Construct object
  R <- list(
    "epsilon" = epsilon,    # Risk of overestimated significance
    "sample_G" = sample_G,  # MC sampler
    "S" = S,                # Number of observed ones from MC sampler
    "n" = n,                # Total number of samples drawn from MC sampler
    "ptilde" = ptilde,      # Sequential estimate of the p-value
    "Ltilde" = Ltilde,      # Sequential lower confidence sequence limit at significance level epsilon
    "B" = B,                # (redundant) sequence of observed ones from MC sampler
    "Bn" = Bn               # (redundant) sequence of total number of samples drawn at each point
  )
  class(R) <- "avseqmc_progress"
  return(R)
}
