#' Treatments
#' 
#' This dataset contains the rearing tank and respective treatment assigned to 
#' it. It represents the metadata from a feeding trial with Atlantic salmon 
#' (Salmo salar). The related dataset was published by Liland et al. (2024) 
#' and is from the first out of two trials ("Trial A"). 
#' 
#' @format A tibble with nine rows and two columns:
#' \describe{
#'  \item{tank}{factor; rearing tank identifier.}
#'  \item{treatment}{factor; treatment identifier.}
#' }
#' 
#' @examples
#' treatments
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name treatments
NULL

#' Daily water parameters
#' 
#' This dataset contains daily readings of water temperature, salinity, and 
#' dissolved oxygen from a feeding trial with Atlantic salmon (Salmo salar).
#' The original dataset ("WaterParametersDaily") was published by
#' Liland et al. (2024) and is from the first out of two trials 
#' ("Trial A"). Alterations to the original data structure were done by 
#' 1) converting the double-row column names into single-row column names,
#' 2) shortening the column names, and 3) converting the table into long 
#' format by moving the tank IDs into a separate column.
#' 
#' @format A tibble with 270 rows and 6 columns:
#' \describe{
#'  \item{date}{date; date of data recording.}
#'  \item{tank}{factor; rearing tank identifier.}
#'  \item{temp}{numeric; water temperature in degrees Celsius.}
#'  \item{salinity}{numeric; salinity in parts per thousand.}
#'  \item{do_perc}{numeric; dissolved oxygen in percentage saturation.}
#'  \item{do_conc}{numeric; dissolved oxygen concentration in mg/L.}
#' }
#'
#' @examples
#' water_params
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name water_params
NULL





#' Daily feed intake
#' 
#' This dataset contains daily feed intake data from a feeding trial with 
#' Atlantic salmon (Salmo salar).
#' The original dataset ("FeedIntake") was published by Liland et al. (2024) 
#' and is from the first out of two trials ("Trial A"). Alterations to the 
#' original data structure were done by 1) converting the double-row column 
#' names into single-row column names, 2) shortening the column names, and 3) 
#' converting the table into long format by moving the tank IDs into a separate 
#' column. NA values have been replaced by the recalculated feed intakes for 
#' the respective day.
#' 
#' @format A tibble with 270 rows and 4 columns:
#' \describe{
#'  \item{date}{date; date of data recording.}
#'  \item{tank}{factor; rearing tank identifier.}
#'  \item{daily_feed_intake}{numeric; daily feed intake in gram per fish.}
#'  \item{cumulative_feed_intake}{numeric; cumulatic feed intake in gram per 
#'  fish.}
#' }
#' 
#' @examples
#' feed_intake
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name feed_intake
NULL





#' Sampling data
#' 
#' This dataset contains the fish weight, fork length, sex, liver weight, 
#' hepatosomatic index, heart weight, cardiosomatic index, gonad weight, 
#' gonadosomatic index, viscera weight, and viscerosomatic index from a 
#' feeding trial with Atlantic salmon (Salmo salar).
#' The original dataset ('FeedIntake') was published by Liland et al. (2024)
#' and is from the first out of two trials ('Trial A'). Alterations to the 
#' original data structure were done by 1) converting the double-row column 
#' names into single-row column names, and 2) removing the 'sex' column.
#' 
#' @format A tibble with 181 rows and 14 columns:
#' \describe{
#'  \item{date}{date; date of data recording.}
#'  \item{tank}{factor; rearing tank identifier.}
#'  \item{replicate}{factor; replicate identifier.}
#'  \item{sample_type}{factor; type of sample.}
#'  \item{fish_weight}{numeric; weight of fish in gram.}
#'  \item{fork_length}{numeric; length of fish from the tip of the snout to the 
#'  end of the middle caudal fin rays in centimeter.}
#'  \item{liver_weight}{numeric; weight of the liver in gram.}
#'  \item{hsi}{numeric; hepatosomatic index in percent.}
#'  \item{heart_weight}{numeric; weight of the heart in gram.}
#'  \item{csi}{numeric; cardiosomatic index in percent.}
#'  \item{gonad_weight}{numeric; weight of the gonads in gram.}
#'  \item{gsi}{numeric; gonadosomatic index in percent.}
#'  \item{viscera_weight}{numeric; weight of the viscera in gram.}
#'  \item{vsi}{numeric; viscerosomatic index in percent.}
#' }
#' 
#' @examples
#' samplings
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name samplings
NULL





#' Daily count of live and dead fish
#' 
#' This dataset contains the daily count of dead and alive fish during a 29-day
#' feeding trial with Atlantic salmon (Salmo salar).
#' The original dataset ('NumberOfFish') was published by Liland et al. (2024)
#' and is from the first out of two trials ('Trial A'). Alterations to the 
#' original data structure were done by 1) converting the double-row column 
#' names into single-row column names, 2) shortening the column names, and 3) 
#' converting the table into long format by moving the tank IDs into a separate 
#' column.
#' 
#' @format A tibble with 270 rows and 4 columns:
#' \describe{
#'  \item{date}{date; date of data recording.}
#'  \item{tank}{factor; rearing tank identifier.}
#'  \item{total_number}{numeric; count of fish stocked at the beginning of the 
#'  experiment.}
#'  \item{dead}{numeric; count of dead fish on each recorded day of the 
#'  experiment.}
#' }
#' 
#' @examples
#' fishcount
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name fishcount
NULL





#' Fish body composition
#' 
#' This dataset contains compositional data on the proximate composition and
#' some key minerals in the body of Atlantic salmon (Salmo salar) at the 
#' beginning and the end of a trial.
#' The original dataset ('BodyComposition_tank') was published by Liland et al. 
#' (2024) and is from the first out of two trials ('Trial A'). Alterations to 
#' the original data structure were done by 1) converting the double-row column 
#' names into single-row column names and 2) removing the 'sex' column.
#' 
#' @format A tibble with 12 rows and 14 columns:
#' \describe{
#'  \item{date}{date; date of data recording.}
#'  \item{treatment}{factor; treatment identifier.}
#'  \item{tank}{factor; rearing tank identifier.}
#'  \item{dm}{numeric; dry matter content of fish tissue in g/g.}
#'  \item{water}{numeric; content of fish tissue in g/g.}
#'  \item{ash}{numeric; content of fish tissue in g/g on dry matter 
#'  basis.}
#'  \item{energy}{numeric; gross energy content of fish tissue in kJ/g on dry 
#'  matter basis.}
#'  \item{fat}{numeric; crude fat content of fish tissue in g/g on dry 
#'  matter basis.}
#'  \item{protein}{numeric; crude protein content of fish tissue in g/g on
#'  dry matter basis.}
#'  \item{ca}{numeric; calcium content of fish tissue in mg/kg on dry matter 
#'  basis.}
#'  \item{k}{numeric; potassium content of fish tissue in mg/kg on dry matter 
#'  basis.}
#'  \item{mg}{numeric; magnesium content of fish tissue in mg/kg on dry matter 
#'  basis.}
#'  \item{na}{numeric; sodium content of fish tissue in mg/kg on dry matter 
#'  basis.}
#'  \item{phosphorus}{numeric; content of fish tissue in mg/kg on dry matter 
#'  basis.}
#' } 
#' 
#' @examples
#' bodycomp
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name bodycomp
NULL





#' Feed composition
#' 
#' This dataset contains compositional data of Skretting Protec, a commercial
#' fish feed. The data comprises the proximate composition, phosphorus, and
#' some essential amino acids. 
#' 
#' @format A tibble with one row and 25 columns:
#' \describe{
#'  \item{diet}{character; feed name.}
#'  \item{dry_matter}{numeric; content of feed in g/g.}
#'  \item{crude_protein}{numeric; content of feed in g/g as fed.}
#'  \item{crude_lipids}{numeric; content of feed in g/g as fed.}
#'  \item{ash}{numeric; content of feed in g/g as fed.}
#'  \item{gross_energy}{numeric; content of feed in MJ per kg as fed.}
#'  \item{phosphorus}{numeric; content of feed in g/g as fed.}
#'  \item{arginine}{numeric; content of feed in g/g as fed.}
#'  \item{histidine}{numeric; content of feed in g/g as fed.}
#'  \item{isoleucine}{numeric; content of feed in g/g as fed.}
#'  \item{leucine}{numeric; content of feed in g/g as fed.} 
#'  \item{lysine}{numeric; content of feed in g/g as fed.}
#'  \item{threonine}{numeric; content of feed in g/g as fed.}
#'  \item{tryptophan}{numeric; content of feed in g/g as fed.}
#'  \item{valine}{numeric; content of feed in g/g as fed.}
#'  \item{methionine}{numeric; content of feed in g/g as fed.}
#'  \item{cysteine}{numeric; content of feed in g/g as fed.}
#'  \item{phenylalanine}{numeric; content of feed in g/g as fed.}
#'  \item{tyrosine}{numeric; content of feed in g/g as fed.}
#'  \item{aspartic_acid}{numeric; content of feed in g/g as fed.}
#'  \item{glutamic_acid}{numeric; content of feed in g/g as fed.}
#'  \item{alanine}{numeric; content of feed in g/g as fed.}
#'  \item{glycine}{numeric; content of feed in g/g as fed.}
#'  \item{proline}{numeric; content of feed in g/g as fed.}
#'  \item{serine}{numeric; content of feed in g/g as fed.}
#' }
#' 
#' @examples
#' feedcomp
#' 
#' @references Liland, N., Rønnestad, I., Azevedo, M., Lai, F., Oulie, F., 
#' Conceição, L., Soares, F. (2024): Dataset on the performance of Atlantic 
#' salmon (Salmo salar) reared at different dissolved oxygen levels under 
#' experimental conditions. Data in Brief 57, 110983. 
#' https://doi.org/10.1016/j.dib.2024.110983
#' 
#' @docType data
#' @keywords datasets
#' @name feedcomp
NULL





#' Digestibility of dry matter
#' 
#' This dataset contains example data to calculate the apparent digestibility
#' of two diets that differ in their dry matter mass fraction and in the mass 
#' fraction of digestibility markers in the feces.
#' 
#' @format A tibble with two rows and four columns:
#' \describe{
#'  \item{diet}{character; feed identifier.}
#'  \item{dm}{numeric; mass fraction of dry matter in the feed. Data reported 
#'  as value between 0 and 1, corresponding to g/g.}
#'  \item{std_feed}{numeric; mass fraction of digestibility marker in feed. 
#'  Data reported as value between 0 and 1, corresponding to g/g.}
#'  \item{std_feces}{numeric; mass fraction of digestibility marker in feces.
#'  Data reported as value between 0 and 1, corresponding to g/g.}
#' }
#' 
#' @examples
#' digestdm
#' 
#' @docType data
#' @keywords datasets
#' @name digestdm
NULL





#' Digestibility of feed nutrients
#' 
#' This dataset contains example data for the calculation of apparent 
#' digestibility coefficients for the digestibility of nitrogen in the dry 
#' matter fraction of an aquaculture feed.
#' 
#' @format A tibble with two rows and six columns:
#' \describe{
#'  \item{diet}{character; feed identifier.}
#'  \item{dm}{numeric; mass fraction of dry matter in the feed. Data reported 
#'  as value between 0 and 1, corresponding to g/g.}
#'  \item{N_feed}{numeric; mass fraction of nitrogen in feed. Data reported as 
#'  value between 0 and 1, corresponding to g/g.}
#'  \item{std_feed}{numeric; mass fraction of digestibility marker in feed.
#'  Data reported as value between 0 and 1, corresponding to g/g.}
#'  \item{N_feces}{numeric; mass fraction of nitrogen in feces in g/kg.}
#'  \item{std_feces}{numeric; mass fraction of digestibility marker in feces. 
#'  Data reported as value between 0 and 1, corresponding to g/g.}
#' }
#' 
#' @examples
#' digestnut
#' 
#' @docType data
#' @keywords dataset
#' @name digestnut
NULL





#' Digestibility of a feed ingredient
#' 
#' This dataset contains example data for the determination of the apparent 
#' digestibility of feed ingredients by using a reference diet and replacing 
#' a part (usually 30%) of it by the feed ingredient to be evaluated.
#' The dataset is based on Bureau et al. (1999) but was presented in its 
#' complete form in Bureau & Hua (2006).
#' 
#' @format A tibble with one row and eight columns
#' \describe{
#'  \item{diet_reference}{character; reference feed identifier.}
#'  \item{ingredient}{character; feed ingredient identifier.}
#'  \item{adc_reference}{numeric; apparent digestibility coefficient of the 
#'  reference feed. The data is a value between 0 and 1.}
#'  \item{adc_test}{numeric; apparent digestibility coefficient of the test 
#'  feed. The data is a value between 0 and 1.}
#'  \item{dm_reference}{numeric; mass fraction of dry matter in the reference feed.
#'  Data reported as value between 0 and 1, corresponding to g/g.}
#'  \item{dm_ingr}{numeric; mass fraction of dry matter in the feed ingredient. 
#'  Data reported as value between 0 and 1, corresponding to g/g.}
#'  \item{CP_reference}{numeric; mass fraction of crude protein in the pelleted 
#'  reference feed reported as value between 0 and 1, corresponding to g/g.}
#'  \item{CP_ingr}{numeric; mass fraction of crude protein in the feed 
#'  ingredient. Data reported as value between 0 and 1, corresponding to g/g.}
#' }
#' 
#' @examples
#' digestingr
#' 
#' @references Bureau, D. P. & Hua, K. (2006). Letter to the Editor of 
#' Aquaculture. Aquaculture, 252(2-4), 103–105. 
#' https://doi.org/10.1016/j.aquaculture.2006.01.028
#' @references Bureau, D. P., Harris, A. M. & Cho, C. Y. (1999). Apparent 
#' digestibility of rendered animal protein ingredients for rainbow trout 
#' (Oncorhynchus mykiss). Aquaculture, 180, 345–358. 
#' https://doi.org/10.1016/S0044-8486(99)00210-0
#' 
#' @docType data
#' @keywords dataset
#' @name digestingr
NULL




#' Fish biomass and bodyweights from the first polyculture experiment
#' 
#' A dataset containing fish weights from a polyculture experiment with 
#' pikeperch (Sander lucioperca) and Russian sturgeon.
#' 
#' @format A tibble with 24 rows and eight columns:
#' \describe{
#'  \item{tank}{character; rearing tank identifier.}
#'  \item{treat_id}{character; treatment identifier.}
#'  \item{count_start}{numeric; fish count at the beginning of the experiment.}
#'  \item{count_end}{numeric; fish count at the end of the experiment.}
#'  \item{biomass_kg_start}{numeric; total fish biomass in kg at the beginning
#'  of the experiment.}
#'  \item{biomass_kg_end}{numeric; total fish biomass in kg at the end of the 
#'  experiment.}
#' }
#' 
#' @examples
#' weight
#' 
#' @docType data
#' @keywords datasets
#' @name weight
NULL





#' Individual fish bodyweights during a polyculture experiment
#' 
#' A dataset holding weight data of tagged individuals of two different fish 
#' species, pikeperch (Sander lucioperca) and largemouth bass (Micropterus 
#' salmoides), at the beginning and the end of a polyculture experiment for 
#' aquaculture purposes.
#' 
#' @format A tibble with 396 rows and four columns:
#' \describe{
#'  \item{species}{character; species identifier.}
#'  \item{culture_sys}{character; culture system identifier.}
#'  \item{ibw_g}{numeric; initial bodyweight in gram.}
#'  \item{fbw_g}{numeric; final bodyweight in gram.}
#' }
#' 
#' @examples
#' weight2
#'
#' @references Pěnka T., Malinovskyi O., Křišťan J., Imentai A., Policar T. 
#' (2021): Effect of density and mixed culture with pikeperch (Sander 
#' lucioperca) on effectivity of largemouth bass (Micropterus salmoides) 
#' intensive culture. Czech Journal of Animal Science, 66: 428–440.
#' 
#' @docType data
#' @keywords datasets
#' @name weight2
NULL





#' Rearing data from the first polyculture experiment
#' 
#' A dataset containing feed quantities fed per tank and the average water 
#' temperature over all tanks stocked with pikeperch oder sturgeon. The 
#' variables are as follows:
#' 
#' @format A data frame containing 24 rows and four columns:
#' \describe{
#'    \item{tank}{character; unique identifier for each fish tank.}
#'    \item{feed_g}{numeric; total feed input in gram over the duration of the 
#'    experiment.}
#'    \item{temp_C}{numeric; average water temperature in degrees Celsius.}
#'    \item{duration}{numeric; total duration of the experiment in days.}
#' }
#' 
#' @examples
#' rearing
#' 
#' @docType data
#' @keywords datasets
#' @name rearing
NULL
