/*
  Author: Xuye Luo
  Date: December 12, 2025
*/

#include <Rcpp.h>
#include <math.h>
using namespace Rcpp;

// [[Rcpp::interfaces(r, cpp)]]

/* 
//' @title Calculate Upsilon Goodness-of-Fit Statistic (C++ Backend)
//' @description Internal C++ function to calculate the statistic.
//' @param x Observed counts (NumericVector).
//' @param p Expected probabilities (NumericVector).
//' @return The calculated statistic.
 */

// [[Rcpp::export]]
double upsilon_gof_statistic_cpp(const NumericVector &x, const NumericVector &p) {
  
  if (x.length() != p.length()) {
    stop("Length of observed counts 'x' and probabilities 'p' must be the same.");
  }

  double N = sum(x);
  
  // Handle empty data case
  if (N == 0) {
    return 0.0;
  }
  
  int k = x.length();
  double statistic = 0.0;
  double expected_N_over_k = N / k;
  
  // If denominator is effectively zero, avoid division by zero
  if (expected_N_over_k < 1e-9) {
    return 0.0;
  }

  for (int i = 0; i < k; i++) {
    double O = x[i];
    double E = N * p[i];
    statistic += pow(O - E, 2) / expected_N_over_k;
  }
  
  return statistic;
}
