#' Compare Multivariate Outlier Detection Methods
#'
#' Compares multiple multivariate outlier detection methods on the same dataset
#'
#' @author Andriy Protsak
#' @param data Input dataset (must be a data.frame)
#' @param methods Vector of method names to compare. Available methods are:
#'        "lof", "dbscan", "knn", "mahalanobis"
#' @param params List of parameters for each method. Must contain named lists:
#'        \itemize{
#'          \item lof: list(K=numeric, threshold=numeric)
#'          \item dbscan: list(max_distance_threshold=numeric, min_pts=numeric)
#'          \item knn: list(d=numeric, K=numeric)
#'          \item mahalanobis: list(alpha=numeric)
#'        }
#'
#' @return None, produces a visualization matrix comparing the outliers detected by each method.
#'
#' @examples
#' inputData = t(matrix(c(3,2,3.5,12,4.7,4.1,5.2,
#' 4.9,7.1,6.1,6.2,5.2,14,5.3),2,7,dimnames=list(c("r","d"))))
#' inputData = data.frame(inputData)
#' methods = c("lof", "dbscan", "knn", "mahalanobis")
#' params = list(
#'   lof = list(K=3, threshold=0.5),
#'   dbscan = list(max_distance_threshold=4, min_pts=3),
#'   knn = list(d=3, K=2),
#'   mahalanobis = list(alpha=0.7)
#' )
#' compare_multivariate_methods(inputData, methods, params)
#'
#' @importFrom graphics par rect text
#'
#' @export
compare_multivariate_methods <- function(data, methods, params) {
  if (!is.data.frame(data)) {
    stop("data must be a dataframe")
  }

  valid_methods <- c("lof", "dbscan", "knn", "mahalanobis")
  if (!all(methods %in% valid_methods)) {
    invalid <- methods[!methods %in% valid_methods]
    stop(sprintf(
      "Invalid multivariate methods: %s",
      paste(invalid, collapse = ", ")
    ))
  }

  # Collect outliers from each method
  outliers_by_method <- list()

  for (method in methods) {
    message("\nExecuting ", method," method")
    outliers_by_method[[method]] <- switch(
      method,
      "lof" = lof(data, params[[method]]$K, params[[method]]$threshold, FALSE),
      "dbscan" = DBSCAN_method(
        data,
        params[[method]]$max_distance_threshold,
        params[[method]]$min_pts,
        FALSE
      ),
      "knn" = knn(data, params[[method]]$d, params[[method]]$K, FALSE),
      "mahalanobis" = mahalanobis_method(data, params[[method]]$alpha, FALSE)
    )
    message("Detected outliers: ", paste(outliers_by_method[[method]], collapse = ", "))

  }

  # Create visualization
  old_par <- par(no.readonly = TRUE)
  on.exit(par(old_par))

  par(mar = c(5, 8, 4, 2))

  plot(
    0,
    0,
    type = "n",
    xlim = c(-0.15, 1),
    ylim = c(0.3, 1),
    xaxt = "n",
    yaxt = "n",
    xlab = "",
    ylab = "",
    main = "Multivariate Outlier Detection Matrix"
  )

  n_points <- nrow(data)
  n_methods <- length(methods)

  cell_width <- 0.7 / n_points
  cell_height <- 0.3 / n_methods

  grid_start_x <- 0.2
  grid_start_y <- 0.8

  # Method names mapping
  method_names <- c(
    "lof" = "LOF",
    "dbscan" = "DBSCAN",
    "knn" = "KNN",
    "mahalanobis" = "Mahalanobis"
  )

  # Draw method labels
  for (i in 1:n_methods) {
    text(
      -0.05,
      grid_start_y - (i - 1) * cell_height - cell_height / 2,
      method_names[methods[i]],
      pos = 4,
      xpd = TRUE,
      cex = 0.8
    )
  }

  # Draw point number labels
  for (i in 1:n_points) {
    text(
      grid_start_x + (i - 1) * cell_width + cell_width / 2,
      grid_start_y + 0.05,
      i,
      xpd = TRUE,
      cex = 0.8
    )
  }

  # Fill grid cells
  for (i in 1:n_methods) {
    for (j in 1:n_points) {
      if (j %in% outliers_by_method[[methods[i]]]) {
        rect(
          grid_start_x + (j - 1) * cell_width,
          grid_start_y - i * cell_height,
          grid_start_x + j * cell_width,
          grid_start_y - (i - 1) * cell_height,
          col = "red",
          border = "white"
        )
      } else {
        rect(
          grid_start_x + (j - 1) * cell_width,
          grid_start_y - i * cell_height,
          grid_start_x + j * cell_width,
          grid_start_y - (i - 1) * cell_height,
          col = "gray90",
          border = "white"
        )
      }
    }
  }
}

#' Compare Univariate Outlier Detection Methods
#'
#' Compares univariate outlier detection methods on the flattened dataset
#'
#' @author Andriy Protsak
#' @param data Input dataset (must be a data.frame)
#' @param methods Vector of method names to compare. Available methods are:
#'        "z_score", "boxandwhiskers"
#' @param params List of parameters for each method. Must contain named lists:
#'        \itemize{
#'          \item z_score: list(d=numeric)
#'          \item boxandwhiskers: list(d=numeric)
#'        }
#'
#' @return None, produces a visualization matrix comparing the outliers detected by each method.
#'
#' @examples
#' inputData = t(matrix(c(3,2,3.5,12,4.7,4.1,5.2,
#' 4.9,7.1,6.1,6.2,5.2,14,5.3),2,7,dimnames=list(c("r","d"))))
#' inputData = data.frame(inputData)
#' methods = c("z_score", "boxandwhiskers")
#' params = list(
#'   z_score = list(d=2),
#'   boxandwhiskers = list(d=2)
#' )
#' compare_univariate_methods(inputData, methods, params)
#'
#' @importFrom graphics par rect text
#'
#' @export
compare_univariate_methods <- function(data, methods, params) {
  if (!is.data.frame(data)) {
    stop("data must be a dataframe")
  }

  valid_methods <- c("z_score", "boxandwhiskers")
  if (!all(methods %in% valid_methods)) {
    invalid <- methods[!methods %in% valid_methods]
    stop("Invalid univariate methods: ",paste(invalid, collapse = ", "))
  }

  message("Executing univariate outlier detection methods:")
  message("Note: Methods are applied to the flattened data vector")

  # Flatten the data into a vector
  flat_vector <- as.vector(as.matrix(data))
  flat_df <- data.frame(x = flat_vector)

  message("\nFlattened vector (length: ", length(flat_vector),")")
  message(paste(round(flat_vector, 2), collapse = " "))

  # Collect outliers from each method
  outliers_by_method <- list()

  for (method in methods) {
    message("\nMethod: ", method)
    outliers_by_method[[method]] <- switch(
      method,
      "z_score" = z_score_method(flat_df, params[[method]]$d, FALSE),
      "boxandwhiskers" = boxandwhiskers(flat_df, params[[method]]$d, FALSE)
    )
    message("Detected outliers: ", paste(outliers_by_method[[method]], collapse = ", "))

  }

  # Create visualization
  old_par <- par(no.readonly = TRUE)
  on.exit(par(old_par))

  par(mar = c(5, 8, 4, 2))

  plot(
    0,
    0,
    type = "n",
    xlim = c(-0.15, 1),
    ylim = c(0.3, 1),
    xaxt = "n",
    yaxt = "n",
    xlab = "",
    ylab = "",
    main = "Univariate Outlier Detection Matrix\n(Flattened Data Vector)"
  )

  n_points <- length(flat_vector)
  n_methods <- length(methods)

  cell_width <- 0.7 / n_points
  cell_height <- 0.3 / n_methods

  grid_start_x <- 0.2
  grid_start_y <- 0.8

  # Method names mapping
  method_names <- c("z_score" = "Z-Score", "boxandwhiskers" = "Box & Whiskers")

  # Draw method labels
  for (i in 1:n_methods) {
    text(
      -0.05,
      grid_start_y - (i - 1) * cell_height - cell_height / 2,
      method_names[methods[i]],
      pos = 4,
      xpd = TRUE,
      cex = 0.8
    )
  }

  # Draw point number labels
  for (i in 1:n_points) {
    text(
      grid_start_x + (i - 1) * cell_width + cell_width / 2,
      grid_start_y + 0.05,
      i,
      xpd = TRUE,
      cex = 0.8
    )
  }

  # Fill grid cells
  for (i in 1:n_methods) {
    for (j in 1:n_points) {
      if (j %in% outliers_by_method[[methods[i]]]) {
        rect(
          grid_start_x + (j - 1) * cell_width,
          grid_start_y - i * cell_height,
          grid_start_x + j * cell_width,
          grid_start_y - (i - 1) * cell_height,
          col = "red",
          border = "white"
        )
      } else {
        rect(
          grid_start_x + (j - 1) * cell_width,
          grid_start_y - i * cell_height,
          grid_start_x + j * cell_width,
          grid_start_y - (i - 1) * cell_height,
          col = "gray90",
          border = "white"
        )
      }
    }
  }
}
