#' @title \eqn{\tau}-value
#' @description Given a game, this function computes its \eqn{\tau}-value.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return The \eqn{\tau}-value of the game, as a vector.
#' @details The \eqn{\tau}-value of \eqn{v\in G^N} is given by
#' \deqn{\tau(v)=m(v)+\alpha (M(v)-m(v)),}
#' where \eqn{M(v)} is the vector of utopia payoffs, \eqn{m(v)} is the vector of minimal rights, and \eqn{\alpha} is the value for which \eqn{\sum_{i\in N}\tau_i(v)=v(N)}.
#' @examples
#' tauvalue(c(0,0,0,0,10,40,30,60,10,20,90,90,90,130,160))
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # cost game
#' -tauvalue(-cost.v) # tau-value of the cost game
#' @seealso \link{minimalrightsvector}, \link{utopiapayoffsvector}.
#' @references Tijs, S. H. (1981). Bounds for the core of a game and the \eqn{\tau}-value. In O. Moeschlin and D. Pallaschke (Eds.), \emph{Game theory and mathematical economics} (pp. 123-132).
#' @export

tauvalue <- function(v, binary=FALSE) {# Principio de la función

  ################################
  ###Comprobación datos entrada###
  ################################

  nC=length(v)#Número de coaliciones.
  n=log(nC+1)/log(2)#Número de jugadores.
  if (n>floor(n))
  {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  #####Creación de variables######
  ################################

  M=c()#Utopia payoffs
  m=c()#Minimal rights.
  tol <- 100*.Machine$double.eps # Precisión para comparar igualdad entre mínimos derechos y pagos de utopía.

  ################################
  ######Cuerpo de la función######
  ################################

  #Utopia Payoffs:
  for (ii in 1:n)
  {
    M[ii] = v[nC]-v[nC-2^(ii-1)] #Mi=v(N)-v(N\ i)
  }

  #Minimal rights:

  m=v[2.^(seq(1:n)-1)] #Vector de v(i).


  for(S in 1:nC)
  {
    num=S #Primero calculo el vector en bits con los jugadores pertenecientes a la coalición en posición S.
    J=floor(log2(num))+1
    Sjug=rep(0,n) # Inicializo la coalición S

    for (ii in 1:J)
    {
      if (num %% 2 ==1)
      {
        Sjug[ii]=1
      }
      num=num %/% 2
    }

    SMii = M*Sjug; # #Pagos de utopia de los jugadores en la coalición.
    for (kk in which(Sjug>0)){#max(v(S)-\sum_{j in S\i}M_j(N,v)))
      m[kk] = max(m[kk], v[S]- sum(SMii)+SMii[kk]);
    }
  }

  if(sum(m>M)>0 || sum(m)>v[nC] || v[nC]>sum(M))
  {
    stop("The core-cover of 'v' is empty, so there is no tau-value.")
  }

  if(abs(sum(M)-sum(m))<tol)
  {#Si los mínimos derechos coinciden con los pagos de utopia, el tau valor coincide con los mínimos derechos.
    tau=m
  }
  else
  {#Calculamos el tau valor
    alpha=(v[nC]-sum(m))/(sum(M)-sum(m));#Calculamos el parámetro alpha por eficiencia.
    tau=m+alpha*(M-m);
  }

  ################################
  ######Salidas de la función#####
  ################################
  return(tau)
}# Fin de la función


