% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_eLOD.R
\name{calc_eLOD}
\alias{calc_eLOD}
\title{Calculate Estimated Limit of Detection (eLOD)}
\usage{
calc_eLOD(data)
}
\arguments{
\item{data}{A \code{soma_adat}, \code{data.frame}, or \code{tibble} object including
SeqId columns (\code{seq.xxxxx.xx}) containing RFU values.}
}
\value{
A \code{tibble} object with 2 columns: SeqId and eLOD.
}
\description{
Calculate the estimated limit of detection (eLOD) for SOMAmer reagent
analytes in the provided input data. The input data should be filtered to
include only buffer samples desired for eLOD calculation.
}
\details{
eLOD is calculated using the following steps:
\enumerate{
\item For each SOMAmer, the median and adjusted median absolute
deviation (\eqn{MAD_{Adjusted}}) are calculated, where
\deqn{MAD_{Adjusted} = 1.4826 * MAD}
The 1.4826 is a set constant used to adjust the MAD to be reflective of
the standard deviation of the normal distribution.
\item For each SOMAmer, calculate \deqn{eLOD = median + 3.3 * MAD_{Adjusted}}
}

Note: The eLOD is useful for non-core matrices, including cell lysate
and CSF, but should be used carefully for evaluating background signal in
plasma and serum.
}
\examples{
# filter data frame using vector of SampleId controls
df <- withr::with_seed(101, {
  data.frame(
    SampleType = rep(c("Sample", "Buffer"), each = 10),
    SampleId = paste0("Sample_", 1:20),
    seq.20.1.100 = runif(20, 1, 100),
    seq.21.1.100 = runif(20, 1, 100),
    seq.22.2.100 = runif(20, 1, 100)
  )
})
sample_ids <- paste0("Sample_", 11:20)
selected_samples <- df |> filter(SampleId \%in\% sample_ids)

selected_elod <- calc_eLOD(selected_samples)
head(selected_elod)
\dontrun{
# filter `soma_adat` object to buffer samples
buffer_samples <- example_data |> filter(SampleType == "Buffer")

# calculate eLOD
buffer_elod <- calc_eLOD(buffer_samples)
head(buffer_elod)

# use eLOD to calculate signal to noise ratio of samples
samples_median <- example_data |> dplyr::filter(SampleType == "Sample") |>
  dplyr::summarise(across(starts_with("seq"), median, .names = "median_{col}")) |>
  tidyr::pivot_longer(starts_with("median_"), names_to = "SeqId",
                      values_to = "median_signal") |>
  dplyr::mutate(SeqId = gsub("median_seq", "seq", SeqId))

# analytes with signal to noise > 2
ratios <- samples_median |>
  dplyr::mutate(signal_to_noise = median_signal / buffer_elod$eLOD) |>
  dplyr::filter(signal_to_noise > 2) |>
  dplyr::arrange(desc(signal_to_noise))

head(ratios)
}
}
\author{
Caleb Scheidel, Christopher Dimapasok
}
