#' Smith and VanderWeele bound
#'
#' `SVbound()` returns a list with the SV bound. All sensitivity parameters for
#' the population of interest must be set to numbers, and the rest can be left
#' as `NULL`. The sensitivity parameters can be inserted directly or as output
#' from `sensitivityparametersM()`. 
#'
#' @param whichEst Input string. Defining the causal estimand of interest.
#'   Available options are as follows. (1) Risk ratio in the total
#'   population: `"RR_tot"`, (2) Risk difference in the total population:
#'   `"RD_tot"`, (3) Risk ratio in the subpopulation: `"RR_sub"`, (4) Risk
#'   difference in the subpopulation: `"RD_sub"`.
#' @param sens Possible method to input sensitivity parameters. `sens` can 
#'   be the output from sensitivityparametersM(), a data.frame with columns 
#'   'parameter' and 'value', or a name list with correct names (e.g. 
#'   `"RR_UY_T1"`,  `"RR_UY_T0"`, etc.). If not supplied, parameters can be 
#'   entered manually as specified below.
#' @param pY1_T1_S1 Input value. The probability P(Y=1|T=1,I_S=1). Must be
#'   between 0 and 1.
#' @param pY1_T0_S1 Input value. The probability P(Y=1|T=0,I_S=1). Must be
#'   between 0 and 1.
#' @param pT1_S1 Input value. The probability P(T=1|I_S=1). Must be
#'   between 0 and 1. Only used for the alternative SV bound for the risk 
#'   difference in the subpopulation. If a value is given to `pT1_S1` and
#'   `pT0_S1`, the alternative bound is used. If they are set to `NULL`, 
#'   the original SV bound will be used.
#' @param pT0_S1 Input value. The probability P(T=0|I_S=1). Must be
#'   between 0 and 1. Only used for the alternative SV bound for the risk 
#'   difference in the subpopulation. If a value is given to `pT1_S1` and
#'   `pT0_S1`, the alternative bound is used. If they are set to `NULL`, 
#'   the original SV bound will be used.
#' @param RR_UY_T1 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_UY|T=1. Must be greater than or equal to 1. 
#'   Used in the bounds for the total population.
#' @param RR_UY_T0 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_UY|T=0. Must be greater than or equal to 1. 
#'   Used in the bounds for the total population.
#' @param RR_SU_11 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_SU|11. Must be greater than or equal to 1. 
#'   Used in the bounds for the total population.
#' @param RR_SU_00 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_SU|00. Must be greater than or equal to 1. 
#'   Used in the bounds for the total population.
#' @param RR_SU_10 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_SU|10. Must be greater than or equal to 1. 
#'   Used in the bounds for the total population.
#' @param RR_SU_01 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_SU|01. Must be greater than or equal to 1. 
#'   Used in the bounds for the total population.
#' @param RR_UY_S1 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_UY|S=1. Must be greater than or equal to 1. 
#'   Used in the bounds for the subpopulation.
#' @param RR_TU_1 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_TU|1. Must be greater than or equal to 1. 
#'   Used in the bounds for the subpopulation.
#' @param RR_TU_0 Possible method to input sensitivity parameter. 
#'   The sensitivity parameter RR_TU|0. Must be greater than or equal to 1. 
#'   Used in the bounds for the subpopulation.
#'
#' @return A list containing the Smith and VanderWeele lower and upper bounds.
#' @export
#'
#' @examples
#' # Example specifying the sensitivity parameters manually. Risk ratio in
#' # the total population.
#' SVbound(whichEst = "RR_tot", pY1_T1_S1 = 0.05, pY1_T0_S1 = 0.01,
#'  RR_UY_T1 = 2, RR_UY_T0 = 2, RR_SU_11 = 1.7, RR_SU_00 = 1.5,
#'  RR_SU_10 = 2.1, RR_SU_01 = 2.3)
#'  
#' # Example specifying the sensitivity parameters manually. Risk difference in
#' # the total population.
#' SVbound(whichEst = "RD_tot", pY1_T1_S1 = 0.05, pY1_T0_S1 = 0.01,
#'  RR_UY_T1 = 2, RR_UY_T0 = 2, RR_SU_11 = 1.7, RR_SU_00 = 1.5,
#'  RR_SU_10 = 2.1, RR_SU_01 = 2.3)
#'
#' # Example specifying the sensitivity parameters manually. Risk ratio in
#' # the subpopulation. 
#' SVbound(whichEst = "RR_sub", pY1_T1_S1 = 0.05, pY1_T0_S1 = 0.01,
#'  RR_UY_S1 = 2.71, RR_TU_1 = 1.91, RR_TU_0 = 2.33)
#'
#' # Example specifying the sensitivity parameters manually. Risk difference in
#' # the subpopulation.
#' SVbound(whichEst = "RD_sub", pY1_T1_S1 = 0.05, pY1_T0_S1 = 0.01,
#'  RR_UY_S1 = 2.71, RR_TU_1 = 1.91, RR_TU_0 = 2.33)
#'  
#' # Example specifying the sensitivity parameters manually. 
#' # Risk difference in the subpopulation with the alternative bound.
#' SVbound(whichEst = "RD_sub", pY1_T1_S1 = 0.05, pY1_T0_S1 = 0.01, pT1_S1 = 0.6,
#' pT0_S1 = 0.3, RR_UY_S1 = 2.71, RR_TU_1 = 1.91, RR_TU_0 = 2.33)
#' 
#' # Example specifying the sensitivity parameters from sensitivityparametersM().
#' # Risk ratio in the subpopulation. DGP from the zika example.
#' V = matrix(c(1, 0, 0.85, 0.15), ncol = 2)
#' U = matrix(c(1, 0, 0.5, 0.5), ncol = 2)
#' Tr = c(-6.2, 1.75)
#' Y = c(-5.2, 5.0, -1.0)
#' S = matrix(c(1.2, 2.2, 0.0, 0.5, 2.0, -2.75, -4.0, 0.0), ncol = 4)
#' probT1 = 0.286
#' probT0 = 0.004
#' senspar = sensitivityparametersM(whichEst = "RR_sub", whichBound = "SV", Vval = V,
#'  Uval = U, Tcoef = Tr, Ycoef = Y, Scoef = S, Mmodel = "L",
#'  pY1_T1_S1 = probT1, pY1_T0_S1 = probT0)
#'  
#' SVbound(whichEst = "RR_sub", sens = senspar, pY1_T1_S1 = probT1, pY1_T0_S1 = probT0)
#'
#' @references  Smith, Louisa H., and Tyler J. VanderWeele. "Bounding bias due
#'   to selection." Epidemiology (Cambridge, Mass.) 30.4 (2019): 509.
#'   
#'   Zetterstrom S, Sjölander A, Waernbaum I. "Investigations of sharp bounds 
#'   for causal effects under selection bias." Statistical Methods in Medical 
#'   Research (2025).
#'
#'   Zetterstrom, Stina and Waernbaum, Ingeborg. "Selection bias and multiple
#'   inclusion criteria in observational studies" Epidemiologic Methods 11, no.
#'   1 (2022): 20220108.
#'
SVbound <- function(whichEst, sens = NULL, pY1_T1_S1, pY1_T0_S1, pT1_S1 = NULL, pT0_S1 = NULL,
                    RR_UY_T1 = NULL, RR_UY_T0 = NULL, RR_SU_11 = NULL,
                    RR_SU_00 = NULL, RR_SU_10 = NULL, RR_SU_01 = NULL,
                    RR_UY_S1 = NULL, RR_TU_1 = NULL, RR_TU_0 = NULL)
{
  
  if (!is.null(sens)) {
    # Case 1: Output from sensitivityparametersM()
    if (inherits(sens, "sensparams")) {
      params <- stats::setNames(as.numeric(sens$value), sens$parameter)
      # Case 2: Data frame with columns 'parameter' and 'value'
    } else if (is.data.frame(sens)) {
      params <- stats::setNames(as.numeric(sens$value), sens$parameter)
      # Case 3: Named list with correct names
    } else if (is.list(sens)) {
      params <- unlist(sens)
    } else {
      stop("'sens' must be a named list (with correct names), data.frame (with columns 'parameter' and 'value'), or 'sensparams' object.")
    }
  }
  
  
  # Check if the estimand is one of the four "RR_tot", "RD_tot", "RR_sub", "RD_sub".
  if(whichEst != "RR_tot" & whichEst != "RD_tot" & whichEst != "RR_sub" & whichEst != "RD_sub")
    stop('The estimand must be "RR_tot", "RD_tot", "RR_sub" or "RD_sub".')

  # Check if the probabilities are valid.
  if((pY1_T1_S1 < 0 | pY1_T1_S1 > 1 | pY1_T0_S1 < 0 | pY1_T0_S1 > 1))
    stop("P(Y=1|T=1,I_S=1) and P(Y=1|T=0,I_S=1) cannot be smaller than 0 or larger than 1.")
  
  # Check if the probabilities are valid.
  if(!is.null(pT1_S1) & !is.null(pT0_S1))
  {
    if((pT1_S1 > 1 | pT0_S1 > 1 | pT1_S1 < 0 | pT0_S1 < 0))
      stop("P(T=1|I_S=1) and P(T=0|I_S=1) cannot be smaller than 0 or larger than 1 (if not set to NULL).") 
  }

  # The observational estimands.
  RRobs = pY1_T1_S1 / pY1_T0_S1
  RDobs = pY1_T1_S1 - pY1_T0_S1

  # Calculations for the total population.
  if(whichEst == "RR_tot" | whichEst == "RD_tot")
  {
    # Extracting the sensitivity parameters from the input list.
    if (!is.null(sens))
    {
      RR_UY_T1 = params[["RR_UY|T=1"]]
      RR_UY_T0 = params[["RR_UY|T=0"]]
      RR_SU_11 = params[["RR_SU|11"]]
      RR_SU_00 = params[["RR_SU|00"]]
      RR_SU_10 = params[["RR_SU|10"]]
      RR_SU_01 = params[["RR_SU|01"]]
    }
    
    # Check if the correct sensitivity parameters are specified.
    if(is.null(RR_UY_T1) | is.null(RR_UY_T0) | is.null(RR_SU_11) | is.null(RR_SU_00) | is.null(RR_SU_10) | is.null(RR_SU_01))
      stop("When the total population is of interest, RR_UY_T1, RR_UY_T0,
           RR_SU_11, RR_SU_00, RR_SU_10, and RR_SU_01 cannot be equal to NULL." )

    # Check if the sensitivity parameters are valid.
    if(RR_UY_T1 < 1 | RR_UY_T0 < 1 | RR_SU_11 < 1 | RR_SU_00 < 1 | RR_SU_10 < 1 | RR_SU_01 < 1)
      stop("All sensitivity parameters must be greater than or equal to 1.")

    # Calculate BF_11, BF_00, BF_10, and BF_01. Used in both "RR_tot" and "RD_tot".
    BF11 = (RR_UY_T1 * RR_SU_11) / (RR_UY_T1 + RR_SU_11 - 1)
    BF00 = (RR_UY_T0 * RR_SU_00) / (RR_UY_T0 + RR_SU_00 - 1)
    BF10 = (RR_UY_T1 * RR_SU_10) / (RR_UY_T1 + RR_SU_10 - 1)
    BF01 = (RR_UY_T0 * RR_SU_01) / (RR_UY_T0 + RR_SU_01 - 1)

    # SV bound for RR in tot pop.
    lowerboundRRtot = round((RRobs / (BF11 * BF00)), 2)
    upperboundRRtot = round((RRobs * (BF10 * BF01)), 2)


    # SV bound for RD in tot pop.
    lowerboundRDtot = round((RDobs - (BF11 - pY1_T1_S1 / BF11 + pY1_T0_S1 * BF00)), 2)
    upperboundRDtot = round((RDobs + (BF01 - pY1_T0_S1 / BF01 + pY1_T1_S1 * BF10)), 2)

  
  }else
  { # Calculations for the subpopulation.
    if (!is.null(sens)) 
    {
      # Extracting the sensitivity parameters from the input list.
      RR_UY_S1 = params[["RR_UY|S=1"]]
      RR_TU_1 = params[["RR_TU|1"]]
      RR_TU_0 = params[["RR_TU|0"]]
    }
    
    # Check if the correct sensitivity parameters are specified.
    if(is.null(RR_UY_S1) | is.null(RR_TU_1 ) | is.null(RR_TU_0))
      stop("When the subpopulation is of interest, RR_UY_S1, RR_TU_1, and RR_TU_0 cannot be equal to NULL." )

    # Check if the sensitivity parameters are valid.
    if(RR_UY_S1 < 1 | RR_TU_1 < 1 | RR_TU_0 < 1)
      stop("All sensitivity parameters must be greater than or equal to 1.")

    # Calculate BF_1. Used in both "RR_sub" and "RD_sub".
    BF1 = (RR_UY_S1 * RR_TU_1) / (RR_UY_S1 + RR_TU_1 - 1)
    BF0 = (RR_UY_S1 * RR_TU_0) / (RR_UY_S1 + RR_TU_0 - 1)

    # SV bound for RR in subpop.
    lowerboundRRs = round(RRobs / BF1, 2)
    upperboundRRs = round(RRobs * BF0, 2)
    
    # Check if the alternative SV bound is calculated.
    if(!is.null(pT1_S1) & !is.null(pT0_S1))
    {
      # Alternative SV bound for RD in subpop.
      lowerboundRDs = round((RDobs - (pY1_T0_S1 * pT1_S1 * (BF1 - 1) + pY1_T1_S1 * pT0_S1 * (1 - 1 / BF1))), 2)
      upperboundRDs = round((RDobs + (pY1_T1_S1 * pT0_S1 * (BF0 - 1) + pY1_T0_S1 * pT1_S1 * (1 - 1 / BF0))), 2)
    } else{
      # SV bound for RD in subpop.
      lowerboundRDs = round((RDobs - max((pY1_T0_S1 * (BF1 - 1)), (pY1_T1_S1 * (1 - 1 / BF1)))), 2)
      upperboundRDs = round((RDobs + max((pY1_T1_S1 * (BF0 - 1)), (pY1_T0_S1 * (1 - 1 / BF0)))), 2)
    }
  }
  
  if(!is.null(pT1_S1) & !is.null(pT0_S1))
  {
    # The return list.
    heading = c("Alternative SV lower bound", "Alternative SV upper bound")
  } else{
    # The return list.
    heading = c("SV lower bound", "SV upper bound")
  }

  if(whichEst == "RR_tot"){
    values = list(lowerboundRRtot, upperboundRRtot)
  }else if(whichEst == "RD_tot"){
    values = list(lowerboundRDtot,upperboundRDtot)
  }else if(whichEst == "RR_sub"){
    values = list(lowerboundRRs, upperboundRRs)
  }else{
    values = list(lowerboundRDs, upperboundRDs)
  }

  returnDat = matrix(cbind(heading, values), ncol = 2)
  return(returnDat)


}
