#' @keywords internal
"_PACKAGE"

#' @title SVG: Spatially Variable Genes Detection Methods
#'
#' @description
#' A unified framework for detecting spatially variable genes (SVGs) in spatial
#' transcriptomics data. This package integrates multiple state-of-the-art SVG
#' detection methods:
#'
#' \itemize{
#'   \item \strong{MERINGUE}: Moran's I with binary adjacency network
#'   \item \strong{Seurat}: Moran's I with inverse distance weights
#'   \item \strong{binSpect}: Binary spatial enrichment test (from Giotto)
#'   \item \strong{SPARK-X}: Non-parametric kernel-based test
#'   \item \strong{nnSVG}: Nearest-neighbor Gaussian processes
#'   \item \strong{MarkVario}: Mark variogram (from spatstat)
#' }
#'
#' @section Main Functions:
#' \itemize{
#'   \item \code{\link{CalSVG}}: Unified interface for all SVG methods
#'   \item \code{\link{CalSVG_MERINGUE}}: MERINGUE method (Moran's I with network)
#'   \item \code{\link{CalSVG_Seurat}}: Seurat method (Moran's I with 1/d^2 weights)
#'   \item \code{\link{CalSVG_binSpect}}: Giotto binSpect method
#'   \item \code{\link{CalSVG_SPARKX}}: SPARK-X method
#'   \item \code{\link{CalSVG_nnSVG}}: nnSVG method (requires BRISC)
#'   \item \code{\link{CalSVG_MarkVario}}: Mark variogram method
#' }
#'
#' @section Utility Functions:
#' \itemize{
#'   \item \code{\link{buildSpatialNetwork}}: Build spatial neighborhood network
#'   \item \code{\link{moranI}}: Calculate Moran's I statistic
#'   \item \code{\link{binarize_expression}}: Binarize gene expression
#' }
#'
#' @author Zaoqu Liu \email{liuzaoqu@@163.com}
#'
#' @references
#' \itemize{
#'   \item Miller, B.F. et al. (2022) nnSVG for spatial transcriptomics. Nature Communications.
#'   \item Sun, S. et al. (2020) Statistical analysis of spatial expression patterns. Nature Methods.
#'   \item Dries, R. et al. (2021) Giotto: a toolbox for spatial transcriptomics. Genome Biology.
#'   \item Miller, J.A. et al. (2021) MERINGUE: characterizing spatial gene expression. Genome Research.
#' }
#'
#' @useDynLib SVG, .registration = TRUE
#' @importFrom Rcpp sourceCpp
#' @importFrom stats dist fisher.test kmeans lm logLik median p.adjust 
#'   pcauchy pchisq pnorm quantile rnbinom runif sd var
#' @importFrom utils setTxtProgressBar txtProgressBar
#' @importFrom MASS ginv
#' @importFrom parallel mclapply detectCores
#' @importFrom methods is
#' @name SVG-package
NULL


#' Unified Interface for SVG Detection
#'
#' @description
#' A unified interface to run different spatially variable gene (SVG) detection
#' methods. This function provides a consistent API for all supported methods.
#'
#' @param expr_matrix Numeric matrix of gene expression values.
#'   Rows are genes, columns are spatial locations (spots/cells).
#'   Should be normalized (e.g., log-transformed counts).
#' @param spatial_coords Numeric matrix of spatial coordinates.
#'   Rows are spatial locations, columns are x and y (and optionally z) coordinates.
#'   Row names should match column names of \code{expr_matrix}.
#' @param method Character string specifying the SVG detection method.
#'   One of: "meringue", "seurat", "binspect", "sparkx", "nnsvg", "markvario".
#' @param n_threads Integer. Number of threads for parallel computation.
#'   Default is 1. Set to higher values for faster computation on multi-core systems.
#' @param verbose Logical. Whether to print progress messages. Default is TRUE.
#' @param ... Additional arguments passed to the specific method function.
#'
#' @return A data.frame containing SVG detection results. The exact columns
#'   depend on the method used, but typically include:
#'   \itemize{
#'     \item \code{gene}: Gene identifiers
#'     \item \code{pval} or \code{p.value}: Raw p-values
#'     \item \code{padj} or \code{p.adj}: Adjusted p-values (multiple testing corrected)
#'     \item Method-specific statistics (e.g., Moran's I, LR statistic, odds ratio)
#'   }
#'
#' @details
#' This function serves as a wrapper around the individual method functions:
#' \itemize{
#'   \item \code{method = "meringue"}: Calls \code{\link{CalSVG_MERINGUE}}
#'   \item \code{method = "seurat"}: Calls \code{\link{CalSVG_Seurat}}
#'   \item \code{method = "binspect"}: Calls \code{\link{CalSVG_binSpect}}
#'   \item \code{method = "sparkx"}: Calls \code{\link{CalSVG_SPARKX}}
#'   \item \code{method = "nnsvg"}: Calls \code{\link{CalSVG_nnSVG}}
#'   \item \code{method = "markvario"}: Calls \code{\link{CalSVG_MarkVario}}
#' }
#'
#' For method-specific parameters, please refer to the documentation of individual
#' method functions.
#'
#' @examples
#' \donttest{
#' # Simulate example data
#' set.seed(42)
#' n_genes <- 20
#' n_spots <- 100
#' expr_matrix <- matrix(rpois(n_genes * n_spots, lambda = 10),
#'                       nrow = n_genes, ncol = n_spots)
#' rownames(expr_matrix) <- paste0("gene_", seq_len(n_genes))
#' colnames(expr_matrix) <- paste0("spot_", seq_len(n_spots))
#'
#' spatial_coords <- cbind(x = runif(n_spots, 0, 100),
#'                         y = runif(n_spots, 0, 100))
#' rownames(spatial_coords) <- colnames(expr_matrix)
#'
#' # Run SPARK-X method (no external dependencies)
#' results <- CalSVG(expr_matrix, spatial_coords, method = "sparkx",
#'                   kernel_option = "single", verbose = FALSE)
#' head(results)
#' }
#'
#' @seealso
#' \code{\link{CalSVG_MERINGUE}}, \code{\link{CalSVG_binSpect}},
#' \code{\link{CalSVG_SPARKX}}, \code{\link{CalSVG_nnSVG}}
#'
#' @export
CalSVG <- function(expr_matrix,
                   spatial_coords,
                   method = c("meringue", "seurat", "binspect", "sparkx", 
                              "nnsvg", "markvario"),
                   n_threads = 1L,
                   verbose = TRUE,
                   ...) {

    # Validate method argument
    method <- match.arg(method)

    # Validate inputs
    if (!is.matrix(expr_matrix) && !inherits(expr_matrix, "Matrix")) {
        expr_matrix <- as.matrix(expr_matrix)
    }

    if (!is.matrix(spatial_coords)) {
        spatial_coords <- as.matrix(spatial_coords)
    }

    # Ensure sample names match
    if (is.null(colnames(expr_matrix))) {
        colnames(expr_matrix) <- paste0("spot_", seq_len(ncol(expr_matrix)))
    }
    if (is.null(rownames(spatial_coords))) {
        rownames(spatial_coords) <- colnames(expr_matrix)
    }

    # Find common samples
    common_samples <- intersect(colnames(expr_matrix), rownames(spatial_coords))
    if (length(common_samples) == 0) {
        stop("No matching samples between expr_matrix columns and spatial_coords rows.")
    }

    if (length(common_samples) < ncol(expr_matrix)) {
        warning(sprintf("Only %d of %d samples have matching coordinates.",
                       length(common_samples), ncol(expr_matrix)))
    }

    # Subset to common samples
    expr_matrix <- expr_matrix[, common_samples, drop = FALSE]
    spatial_coords <- spatial_coords[common_samples, , drop = FALSE]

    if (verbose) {
        message(sprintf("Running SVG detection with method: %s", toupper(method)))
        message(sprintf("  Number of genes: %d", nrow(expr_matrix)))
        message(sprintf("  Number of spots: %d", ncol(expr_matrix)))
    }

    # Dispatch to specific method
    result <- switch(method,
        "meringue"  = CalSVG_MERINGUE(expr_matrix, spatial_coords,
                                      n_threads = n_threads, verbose = verbose, ...),
        "seurat"    = CalSVG_Seurat(expr_matrix, spatial_coords,
                                    n_threads = n_threads, verbose = verbose, ...),
        "binspect"  = CalSVG_binSpect(expr_matrix, spatial_coords,
                                      n_threads = n_threads, verbose = verbose, ...),
        "sparkx"    = CalSVG_SPARKX(expr_matrix, spatial_coords,
                                    n_threads = n_threads, verbose = verbose, ...),
        "nnsvg"     = CalSVG_nnSVG(expr_matrix, spatial_coords,
                                   n_threads = n_threads, verbose = verbose, ...),
        "markvario" = CalSVG_MarkVario(expr_matrix, spatial_coords,
                                       n_threads = n_threads, verbose = verbose, ...)
    )

    return(result)
}
