
#' @title Dimensionless Distribution Coefficient

#' @description This function computes the natural logarithm of dimensionless distribution coefficient (Kd) from adsorption equilibrium data across temperatures, and non-temperature treatments, if any. This function prepares the foundational input required for thermodynamic parameter analysis.



#' @param V Volumn of water in litre
#' @param m Mass of the adsorbent in gram
#' @param MW.Ad Molar weight of the adsorbate in gram per mole
#' @param M.Ad Molarity of the solvent in mole per litre(55.5 for water)
#' @param Non_T_trt Non-temperature treatment
#' @param T_trt Temperature treatment
#' @param Rep Replication
#' @param IGC Initial graded concentrations of the adsorbate in milligram per litre
#' @param Ce Equilibrium concentration of the adsorbate in milligram per litre


#' @return
#' \itemize{
#'   \item qe_i: Absorbed amount in milligram per kg
#'   \item Kd_i: Distribution coefficient
#'   \item lnKd_i: Natural logarithm of Kd
#'   \item Mean_lnkd: Average of lnKd
#' }

#' @export
#' @importFrom dplyr group_by summarise across where
#' @importFrom magrittr %>%
#' @usage LnKd(V, m, MW.Ad, M.Ad, Non_T_trt, T_trt, Rep, IGC, Ce)
#' @examples
#' V <- 0.02  # in litre
#' m <- 2 # in gram
#' MW.Ad <- 75
#' M.Ad <- 55.5
#' Non_T_trt <- c(0,0,0,0,0,0,1,1,1,1,1,1)
#' T_trt <- c(1,1,1,2,2,2,1,1,1,2,2,2)
#' Rep <- c(1,2,3,1,2,3,1,2,3,1,2,3)
#' IGC <- c(2,4)
#' Ce2 <- c(0.030, 0.031, 0.032, 0.033, 0.034, 0.035, 0.030, 0.031, 0.032, 0.033, 0.034, 0.035)
#' Ce4 <- c(0.030, 0.031, 0.032, 0.033, 0.034, 0.035, 0.030, 0.031, 0.032, 0.033, 0.034, 0.035)
#' Ce <- data.frame(Ce2, Ce4)
#' my.LnKd<- LnKd(V, m, MW.Ad, M.Ad, Non_T_trt, T_trt, Rep, IGC, Ce)




#' @references Roy, A., Manjaiah, K. M., Datta, S. P., Rakshit, D., Barman, M., Ray, P., Golui, D., Raza, M. B., Tigga, P., Mondal, S., Vishwanath, Meena, S., & Meena, P. (2025). Effect of Low-Molecular-Weight Organic Acids and Silicon on Arsenic Adsorption and Desorption in a Paddy Soil of Bengal Delta Plain: Insights from Thermodynamics and Equilibrium Modeling. Water, Air, & Soil Pollution, 236(6), 344. https://doi.org/10.1007/s11270-025-07963-7


LnKd <- function(V, m, MW.Ad, M.Ad, Non_T_trt, T_trt, Rep, IGC, Ce) {

  IGC_matrix <- matrix(IGC, nrow = nrow(Ce), ncol = ncol(Ce), byrow = TRUE)

  qei <- (IGC_matrix - Ce)*(V/m)

  Kdi <- ((IGC_matrix - Ce)/Ce)*(V/m)*MW.Ad*M.Ad

  lnKdi <- log(Kdi)

  qe_out <- data.frame(Non_T_trt, T_trt, Rep, qei)

  Kdi_out <- data.frame(Non_T_trt, T_trt, Rep, Kdi)

  lnKdi_out <- data.frame(Non_T_trt, T_trt, Rep, lnKdi)

  mean_table <- lnKdi_out %>%
    dplyr::group_by(Non_T_trt, T_trt) %>%
    dplyr::summarise(dplyr::across(where(is.numeric), ~ mean(.x, na.rm = TRUE)),
              .groups = "drop")

  Avearge_mean_table <- (as.data.frame(mean_table[,-3]))

  output<- list(qe_i = qe_out, Kd_i = Kdi_out, lnKd_i = lnKdi_out,
                Mean_lnkd = Avearge_mean_table)

  return(output)
}










#' @title Estimation of Slope and Intercept

#' @description Generates slope and intercept values from temperature-dependent lnKd data using linear regression, and the corresponding coefficient of determination (\eqn{R^2}) values. These coefficients form the basis for calculating the thermodynamic parameters, providing a simple and transparent bridge between experimental equilibrium measurements and thermodynamic interpretation.

#' @param lnKd Natural logarithm of distribution coefficient
#' @param Temp Temperature in Kelvin

#' @return
#' \itemize{
#'   \item Intercept: Intercept of the fitted line
#'   \item Slope: Slope of the fitted line
#'   \item R_square: Coefficient of determination of the fitted line
#' }

#' @export
#' @import stats
#' @usage Slope_Intercept(lnKd, Temp)
#' @examples
#' lnKd <- c(5.01, 5.02)
#' Temp <- c (298, 303)
#' my.SI<- Slope_Intercept(lnKd, Temp)

#' @references Gouaich, I., Bestani, B., Bouberka, Z., Srenscek-Nazza, J., Michalkiewicz, B., Benzekri-Benallou, M., Boucherdoud, A., and Benderdouche, N. (2023). Characterization of a low-cost Eucalyptus camaldulensis leaves based activated carbon for pharmaceutical residues removal from aqueous solutions. Desalination and Water Treatment, 296, 19–31. https://doi.org/10.5004/dwt.2023.29602

Slope_Intercept <- function(lnKd, Temp){

  f.model <- stats::lm(lnKd ~ I(1/Temp))

  intercept <- unname(stats::coef(f.model)[1])
  slope <- unname(stats::coef(f.model)[2])
  R2 <- summary(f.model)$r.squared

  output <- list(Intercept = intercept, Slope = slope, R_square = R2)
  return(output)

}









#' @title Estimation of Thermodynamic Parameters

#' @description Calculates delta H, delta S, and delta G across temperature and non-temperature treatments using regression-derived slope and intercept values, integrating van’t Hoff and Gibbs-based relationships. This function assesses spontaneity, energetic favourability, and system randomness, providing a comprehensive thermodynamic profile for interpreting adsorption energetics.


#' @param lnKd Natural logarithm of distribution coefficient corresponding to each initial graded concentrations
#' @param Temp Temperature in Kelvin
#' @param Slope Estimated slope of the fitted line
#' @param Intercept Estimated intercept of the fitted line

#' @return
#' \itemize{
#'   \item Delta_H: Change in the standard enthalpy
#'   \item Delta_S: Change in the standard entropy
#'   \item Delta_G: Change in the standard Gibbs free energy
#'   \item Descriptive: Mean and standard error of the thermodynamic parameters
#' }

#' @export
#' @import stats
#' @usage Thermo_parameters(lnKd, Temp, Slope, Intercept)
#' @examples
#' lnKd <- c(7,8)
#' Temp <- 298
#' Slope <- c(-180, -200)
#' Intercept <- c(5, 6)
#' my.tp <- Thermo_parameters(lnKd, Temp, Slope, Intercept)


#' @references
#' \itemize{

#' \item Roy, A., Manjaiah, K. M., Datta, S. P., Rakshit, D., Barman, M., Ray, P., Golui, D., Raza, M. B., Tigga, P., Mondal, S., Vishwanath, Meena, S., & Meena, P. (2025). Effect of Low-Molecular-Weight Organic Acids and Silicon on Arsenic Adsorption and Desorption in a Paddy Soil of Bengal Delta Plain: Insights from Thermodynamics and Equilibrium Modeling. Water, Air, & Soil Pollution, 236(6), 344. https://doi.org/10.1007/s11270-025-07963-7
#' \item Yi, Z., Yao, J., Zhu, M., Chen, H., Wang, F., & Liu, X. (2016). Kinetics, equilibrium, and thermodynamics investigation on the adsorption of lead (II) by coal-based activated carbon. SpringerPlus, 5(1), 1160. https://doi.org/10.1186/s40064-016-2839-4
#' }

Thermo_parameters <- function(lnKd, Temp, Slope, Intercept){
  R <- 0.008314
  Delta_H <- Slope*(-R)
  Delta_S <- Intercept*R
  Delta_G <- -R*Temp*lnKd



  Describe <- matrix(nrow = 2, ncol = 3)
  row.names(Describe) <- c('Mean', 'S. E.')
  colnames(Describe) <- c("Delta_H", "Delta_S", "Delta_G")
  Describe[1,1] <- mean(Delta_H)
  Describe[2,1] <- (stats::sd(Delta_H))/sqrt(length(Delta_H))

  Describe[1,2] <- mean(Delta_S)
  Describe[2,2] <- (stats::sd(Delta_S))/sqrt(length(Delta_S))

  Describe[1,3] <- mean(Delta_G)
  Describe[2,3] <- (stats::sd(Delta_G))/sqrt(length(Delta_G))


  output <- list(Delta = data.frame(Delta_H, Delta_S, Delta_G), Descriptive = Describe)
  return(output)
}
