% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tiger_county_subsection_sf.R
\name{tiger_county_subsection_sf}
\alias{tiger_county_subsection_sf}
\title{tiger_county_subsection_sf}
\usage{
tiger_county_subsection_sf(
  state = NULL,
  output_dir = tempdir(),
  delete_files = TRUE,
  vintage = 2020,
  general = FALSE,
  set_crs = NULL,
  transform_crs = NULL,
  sf_info = FALSE,
  do_progress = FALSE,
  shapefile = NULL,
  datafile = NULL,
  datafile_key = NULL,
  sf_key = "GEOID",
  express = NULL,
  check_na = FALSE
)
}
\arguments{
\item{state}{A required two-digit FIPS code for the state of interest.
See \href{https://cran.r-project.org/package=usmap}{usmap::fips function} for finding FIPS codes.}

\item{output_dir}{A full directory path where the shapefile and its associated files will be downloaded.
The default is the directory defined by the value returned by \code{tempdir()}.}

\item{delete_files}{A logical which if \code{TRUE} will delete the shapefile and associated files in 'output_dir'.
The default is \code{TRUE}.}

\item{vintage}{A numeric that sets the vintage of interest. The default is 2020.}

\item{general}{A logical which if \code{TRUE} will download a less detailed, more generalized version of the tract geometries.}

\item{set_crs}{A numeric or character string which if non-NULL calls sf::st_crs() to set the crs of the geometries and transforms them.}

\item{transform_crs}{A numeric or character string which if non-NULL calls sf::st_transform()
to perform a crs transform of the geometries. Note that the crs of the shapefile must not be NA.}

\item{sf_info}{A logical which if \code{TRUE} displays info on the resulting simple feature object.}

\item{do_progress}{A logical which if \code{TRUE} displays a progress bar during the download.}

\item{shapefile}{A full file path to a shapefile folder with its unzipped files to be processed instead of downloading.}

\item{datafile}{A dataframe containing data that should be joined with this function's resultant simple feature object.}

\item{datafile_key}{The column name from 'datafile' dataframe used to key with the 'sf_key' column of the resultant simple feature dataframe.}

\item{sf_key}{The column from the resultant dataframe used to key with the 'datafile' dataframe.}

\item{express}{A logical expression object used to filter the resultant simple feature dataframe.
For example, one of the columns of the resultant simple feature dataframe is "COUNTYFP".
If you wanted to return just the geometries for Los Alamos, New Mexico (which has a fips code of "028"),
then you assign 'express' equal to: expression(COUNTYFP == "028"). The expression will be
evaluated and only the tract geometries for Los Alamos will be returned.}

\item{check_na}{A logical which if \code{TRUE} will remove rows that have missing values for any of the columns.
The default is to not check the columns for \code{NA} values.}
}
\value{
A data frame object of class sf
}
\description{
This function performs three tasks:
\enumerate{
\item Download to an output directory a zip file from the TIGER/Line Shapefiles database.
\item Unzip the zip file and locate the shape file of interest.
\item Read and convert the shape file to a simple feature object.
}
}
\details{
Returns simple feature (sf) of US Census county subsection boundary related geometric polygons,
provided by the US Census Bureau's TIGER/Line Shapefiles database. See
\href{https://r-spatial.github.io/sf/articles/sf1.html}{Simple Features for R}
for more information on simple features. Along with the geometries, additional county subsection related
variables are provided.  See \href{https://www2.census.gov/geo/pdfs/maps-data/data/tiger/tgrshp2022/TGRSHP2022_TechDoc_F-R.pdf}{Appendix I-3. Record Layout: County Subdivision State-based Shapefile)}
for a description of county subsection related variables of the sf file. For further information on the Census Bureau's shape files see
\href{https://www2.census.gov/geo/pdfs/maps-data/data/tiger/tgrshp2021/TGRSHP2021_TechDoc_Ch3.pdf}{About the 2021 TIGER/Line Shapefiles}.
See \href{https://www2.census.gov/geo/pdfs/maps-data/data/tiger/tgrshp2022/TGRSHP2022_TechDoc_Ch4.pdf}{Chapter 4.8 County Subdivisions} for
a description of the term.

A more generalized, recognizable version of the tract geometries that has less download size is also available.  For more information on cartographic boundary files see
\href{https://www.census.gov/programs-surveys/geography/technical-documentation/naming-convention/cartographic-boundary-file.html}{Cartographic Boundary File Description}.
These files are available for vintages greater than 2013 with resolution 1:500k meters.

The function returns the simple feature object which can easily be mapped (see \href{https://github.com/deandevl/RplotterPkg}{RplotterPkg::create_sf_plot()}) or
joined with US Census Bureau demographic data.  To help incorporate data files, this function
has a 'datafile' parameter which will be joined with the resultant simple feature object. The only
requirement is that a common "key" for joining exist between the data dataframe and the simple feature dataframe.

Some earlier vintages may have \code{NA} for the crs so you may need to specify the 'crs_transform' to 3426.  Also
you may be interested in using a state level crs. See \href{https://epsg.io/}{epsg.io} to search worldwide for crs.
}
\examples{
library(downloader)
library(sf)
library(data.table)
library(withr)
library(usmap)
library(RcensusPkg)

# Find the subsections for a county in Ohio
# Define a temporary, self deleting output folder for the downloaded shapefiles
output_dir <- withr::local_tempdir()
if(!dir.exists(output_dir)){
  dir.create(output_dir)
}

# Define the fips values for state and county
ohio_hc_fips <- usmap::fips(state = "ohio", county = "holmes")
ohio_fips <- substr(ohio_hc_fips,1,2)
hc_fips <- substr(ohio_hc_fips,3,5)

# Define a filtering expression for the county
express <- parse(text = paste0("COUNTYFP == ", '"', hc_fips, '"'))

# Get the subsection sf object for the Ohio county
hc_ctysub_sf <- RcensusPkg::tiger_county_subsection_sf(
  state = ohio_fips,
  vintage = 2020,
  general = TRUE,
  express = express,
  output_dir = output_dir,
  delete_files = FALSE
)

}
