% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VisualizeQuartets.R
\name{VisualizeQuartets}
\alias{VisualizeQuartets}
\title{Visualize quartet difference on trees, by split}
\usage{
VisualizeQuartets(
  tree1,
  tree2,
  style = "pie",
  setPar = TRUE,
  precision = 3L,
  Plot = plot.phylo,
  scale = 1L,
  spectrum = viridisLite::viridis(101),
  legend = TRUE,
  ...
)
}
\arguments{
\item{tree1, tree2}{Trees of class \code{phylo}, with identical leaf labels.}

\item{style}{Character string specifying split labels with an unambiguous
abbreviation of:
\itemize{
\item \code{label}: Label stating proportion of resolved quartets in agreement,
coloured accordingly;
\item \code{pie}: Pie chart showing proportion of quartets in agreement, sized
according to number of quartets influenced by each split;
\item \code{bar}: Bar showing proportion of quartets in agreement, labelled;
\item \code{size}: Circle coloured according to proportion of quartets in agreement,
with area corresponding to number of quartet statements associated with
split.
}}

\item{setPar}{Logical specifying whether graphical parameters should be set
to display trees side by side.}

\item{precision}{Integer specifying number of significant figures to display
when reporting matching scores.}

\item{Plot}{Function to use to plot trees.}

\item{scale}{Numeric, enlargement factor for split labels.}

\item{spectrum}{101-element vector specifying a range of colours by which
to colour matches.}

\item{legend}{Logical specifying whether to display simple legend.}

\item{\dots}{Additional parameters to send to \code{Plot()}.}
}
\value{
\code{VisualizeQuartets()} invisibly returns a list with two elements,
named \code{tree1} and \code{tree2}, containing a matrix.
Each row corresponds to a split within that tree; columns correspond to:
\describe{
\item{node}{The internal numbering of the node corresponding to each split,
as displayed by \code{ape::nodelabels()}}
\item{N, Q, s, d, r1, r2, u}{The status of each quartet relative to that
split, as documented in \code{\link[=QuartetStatus]{QuartetStatus()}}}
\item{res}{The number of quartets resolved by that split, i.e. \code{s} + \code{d}}
\item{same}{The proportion of quartets resolved by that node that are
resolved in the same manner in the other tree; i.e. \code{s / s + d}}
}
}
\description{
Visualize quartet difference on trees, by split
}
\examples{
library("TreeTools", quietly = TRUE)
# Simple plot
VisualizeQuartets(BalancedTree(10), CollapseNode(PectinateTree(10), 19),
                  style = "label")

# Plot with custom graphical parameters
origPar <- par(mfrow = c(2, 2))
VisualizeQuartets(BalancedTree(10), CollapseNode(PectinateTree(10), 19),
                  setPar = FALSE)
VisualizeQuartets(BalancedTree(10), CollapseNode(PectinateTree(10), 19),
                  style = "bar", legend = FALSE, setPar = FALSE)

# Circle size denotes similarity
par(mfrow = c(2, 1), mar = rep(0.1, 4))
vq <- VisualizeQuartets(
  tree1 = BalancedTree(20),
  tree2 = CollapseNode(PectinateTree(20), 29:33),
  style = "size", scale = 2,
  setPar = FALSE # necessary for node labels to align
)
# Manually add custom node labels
percentSame <- paste(round(vq[["tree2"]][, "same"] * 100, 1), "\%")
nodelabels(percentSame, vq[["tree2"]][, "node"],
           frame = "n", bg = NA, # No frame or background
           adj = 0.5 # align label
           )
           
# restore original graphical parameters
par(origPar)
}
\author{
\href{https://orcid.org/0000-0001-5660-1727}{Martin R. Smith}
(\href{mailto:martin.smith@durham.ac.uk}{martin.smith@durham.ac.uk})
}
