#' @title Yield Index (YI)
#' @description
#' This function computes Yield Index (YI) using any traits (like yield) under stress and non-stress conditions. The lower values of YI indicates greater tolerance. For more details see Gavuzzi et al. (1997) <doi:10.4141/P96-130>.
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @param YMS Optional numeric scalar: mean of \code{YS}. If NULL, computed.
#' @return A list with \code{YMS} and \code{Result} (data frame with \code{Gen}, \code{YI}).
#'
#' \deqn{YI = \frac{YS}{YMS}}
#'
#' @references Gavuzzi, P., Rizza, F., Palumbo, M., Campanile, R.G., Ricciardi, G.L. and Borghi, B. (1997). Evaluation of field and laboratory predictors of drought and heat tolerance in winter cereals. Canadian Journal of Plant Science, 77(4), 523-531.
#' @examples
#' out = YI(Gen=c("G1","G2","G3"), YS=c(7,5,3))
#' print(out)
#' @export
YI <- function(Gen, YS, YMS = NULL) {
  YS <- as.numeric(YS)
  if (is.null(YMS)) YMS <- mean(YS, na.rm = TRUE)
  out <- YS / YMS
  out[YMS == 0] <- NA_real_
  Result <- data.frame(Gen = as.character(Gen), YI = out, row.names = NULL)
  return(list(YMS = YMS, Result = Result))
}
