% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/A00-surveff.R
\name{surveff}
\alias{surveff}
\title{Survival Effect Estimation with Propensity Score Weighting}
\usage{
surveff(
  data,
  ps_formula,
  censoring_formula,
  eval_times = NULL,
  weight_method = "IPW",
  att_group = NULL,
  trim = FALSE,
  delta = NULL,
  contrast_matrix = NULL,
  censoring_method = "weibull",
  variance_method = NULL,
  B = 100,
  parallel = FALSE,
  mc.cores = 2,
  seed = NULL,
  censoring_control = NULL,
  ties = "efron",
  ps_control = list(),
  boot_level = "full"
)
}
\arguments{
\item{data}{Data frame containing treatment, outcome, and covariates.}

\item{ps_formula}{Formula for propensity score model: \code{treatment ~ covariates}.}

\item{censoring_formula}{Formula for censoring model: \code{Surv(time, event) ~ covariates}.
Event should be coded as 1=event, 0=censored. Use \code{I(1-event)} if reversed.}

\item{eval_times}{Numeric vector of time points for evaluation. If NULL (default),
uses all unique event times.}

\item{weight_method}{Weighting method: "IPW" (inverse probability weighting),
"OW" (overlap weighting), or "ATT" (average treatment effect on the treated).
Default "IPW".}

\item{att_group}{Target group for ATT. Required if \code{weight_method = "ATT"}.}

\item{trim}{Logical. Perform symmetric propensity score trimming? Default FALSE.
If TRUE, symmetric trimming is applied (Crump extension for multiple treatments).
See \code{\link{estimate_weights}} for trimming details. Ignored if
\code{weight_method = "OW"}. Asymmetric trimming is no longer supported due to
poor statistical performance.}

\item{delta}{Threshold for symmetric trimming in \eqn{(0, 1/J]}, where \eqn{J} is the number
of treatment levels. Default NULL uses recommended values: 0.1 for binary
treatment, 0.067 for 3 groups, \eqn{1/(2J)} for \eqn{J \ge 4} (Yoshida et al., 2019).
Used only if \code{trim = TRUE}.}

\item{contrast_matrix}{Optional matrix for treatment differences in multiple group settings.
Each row defines one contrast with exactly two non-zero elements: -1 and 1.
Column names must match treatment levels. For binary treatment, always estimates
second level minus first level (S1 - S0), ignoring this parameter.}

\item{censoring_method}{Method for censoring score estimation: "weibull" or "cox".
Default "weibull".}

\item{variance_method}{Variance estimation method: "analytical" (binary treatment with
Weibull censoring only) or "bootstrap". Default "analytical" for binary Weibull,
"bootstrap" otherwise. Cox censoring always uses bootstrap.}

\item{B}{Number of bootstrap iterations. Default 100. Used only if \code{variance_method = "bootstrap"}.}

\item{parallel}{Logical. Use parallel bootstrap computation? Default FALSE.}

\item{mc.cores}{Number of cores for parallel bootstrap. Default 2.}

\item{seed}{Random seed for bootstrap reproducibility. Default NULL.}

\item{censoring_control}{Control parameters passed to censoring model fitting function.
For Weibull: passed to \code{survreg()}, default \code{list(maxiter = 350)}.
For Cox: passed to \code{coxph()}, default \code{list()}.}

\item{ties}{Tie handling method for Cox models. Default "efron". Ignored for Weibull.}

\item{ps_control}{Control parameters for propensity score model. Default \code{list()}.}

\item{boot_level}{Bootstrap sampling level: "full" (default) or "strata".
"full" resamples from entire dataset (standard for observational studies). "strata"
resamples within each treatment group preserving group sizes (useful when treatment assignment 
follows a stratified or fixed-ratio design). Only used if \code{variance_method = "bootstrap"}.}
}
\value{
List containing:
  \item{survival_estimates}{Matrix [time x J] of survival function estimates for each group.}
  \item{survival_se}{Matrix [time x J] of standard errors for survival functions.}
  \item{difference_estimates}{Matrix [time x K] of treatment effect estimates.
    For binary treatment: single column with S1-S0. For multiple groups: contrasts
    from \code{contrast_matrix}, or NULL if not provided.}
  \item{difference_se}{Matrix [time x K] of standard errors for treatment effects.}
  \item{eval_times}{Time points evaluated.}
  \item{treatment_levels}{Sorted unique treatment values.}
  \item{n_levels}{Number of treatment groups.}
  \item{n}{Sample size (complete cases after data validation).}
  \item{included}{Logical vector [n] indicating inclusion in analysis. TRUE = included,
    FALSE = excluded due to trimming.}
  \item{estimand}{Estimand used.}
  \item{censoring_method}{Censoring method used.}
  \item{variance_method}{Variance method used.}
  \item{contrast_matrix}{Contrast matrix used (NULL if not applicable).}
  \item{ps_model}{Fitted propensity score model (glm or multinom object).}
  \item{censoring_models}{Named list of fitted censoring models by treatment group.}
  \item{weights}{Numeric vector [n] of final weights (0 for trimmed observations).}
  \item{trim_summary}{Data frame with trimming summary by treatment group.}
  \item{ess}{Named numeric vector of effective sample size by treatment group.}
  \item{boot_result}{Bootstrap results (NULL if analytical variance used).}
}
\description{
Main user interface for estimating counterfactual survival functions and
treatment effects using propensity score weighting and inverse probability
of censoring weighting. Supports binary and multiple treatment groups with
various weighting schemes (IPW, OW, or ATT) and optional trimming.
}
\details{
**Weighting Methods:**

The \code{weight_method} parameter specifies the target population for causal
inference:

\itemize{
  \item \strong{IPW (Inverse Probability Weighting)}: Observations are weighted
    by the inverse probability of their observed treatment, \eqn{w_i = 1/e_j(X_i)}
    where j is the observed treatment group. Inference targets the combined
    population.

  \item \strong{OW (Overlap Weighting)}: Observations are weighted by overlap
    weights, which extends to multiple treatment groups (Li et al., 2018;
    Li and Li, 2019). Inference targets the population at clinical equipoise
    (overlap population).

  \item \strong{ATT (Average Treatment Effect on the Treated)}: IPW weights
    tilted toward a specified target group. Observations in the target group
    receive weight 1, others receive \eqn{w_i = e_{\text{target}}(X_i) / e_j(X_i)}.
    Inference targets the specified treatment group population.
}

**Variance Estimation:**
- Analytical: Binary treatment with Weibull censoring only (M-estimation).
- Bootstrap: All settings (resamples entire pipeline).
- Cox: Always uses bootstrap.

**Treatment Effects:**
- Binary: S1 - S0 (second level minus first).
- Multiple groups: Requires \code{contrast_matrix} for pairwise comparisons.
}
\examples{
\donttest{
# Example 1: Binary treatment with overlap weighting and Weibull censoring model
data(simdata_bin)
result1 <- surveff(
  data = simdata_bin,
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2,
  censoring_formula = survival::Surv(time, event) ~ X1 + B1,
  weight_method = "OW",
  censoring_method = "weibull"
)
summary(result1)
plot(result1)

# Example 2: Multiple treatments with IPW and Cox censoring model
data(simdata_multi)
result2 <- surveff(
  data = simdata_multi,
  ps_formula = Z ~ X1 + X2 + X3 + B1 + B2,
  censoring_formula = survival::Surv(time, event) ~ X1 + B1,
  weight_method = "IPW",
  censoring_method = "cox",
  variance_method = "bootstrap",
  B = 100
)
summary(result2)
}

}
\references{
Li, F., Morgan, K. L., & Zaslavsky, A. M. (2018). Balancing covariates via
propensity score weighting. \emph{Journal of the American Statistical Association},
113(521), 390-400.

Li, F., & Li, F. (2019). Propensity score weighting for causal inference with
multiple treatments. \emph{The Annals of Applied Statistics}, 13(4), 2389-2415.

Yoshida, K., et al. (2019). Multinomial extension of propensity score trimming
methods: A simulation study. \emph{American Journal of Epidemiology}, 188(3),
609-616.

Cheng, C., Li, F., Thomas, L. E., & Li, F. (2022). Addressing extreme propensity
scores in estimating counterfactual survival functions via the overlap weights.
\emph{American Journal of Epidemiology}, 191(6), 1140-1151.
}
