#' Calibrate Softmax Temperature Parameter
#'
#' Selects the optimal temperature parameter for softmax PSRI aggregation
#' using perplexity targeting. The temperature controls how sharply the
#' softmax concentrates weight on the dominant component(s).
#'
#' @param component_profiles A list of numeric vectors, where each vector
#'   represents a set of PSRI component scores (e.g., \code{c(MSG, MRG,
#'   cMTG)}) from representative replicates across treatments.
#' @param target_perplexity Numeric. Target effective number of components.
#'   Default is 2.0, meaning the softmax should effectively use about 2
#'   out of 3 components, concentrating on the dominant signal while
#'   preventing zero-collapse.
#' @param T_range Numeric vector of length 2. Search range for the
#'   temperature parameter. Default is \code{c(0.01, 2.0)}.
#' @param tolerance Numeric. Convergence tolerance for the perplexity
#'   target. Default is 0.05.
#'
#' @return A list with:
#'   \describe{
#'     \item{optimal_T}{The calibrated temperature parameter.}
#'     \item{mean_perplexity}{Mean perplexity across all profiles at the
#'       optimal temperature.}
#'     \item{profile_perplexities}{Named numeric vector of per-profile
#'       perplexities.}
#'     \item{target_perplexity}{The target that was specified.}
#'     \item{search_range}{The \code{T_range} used.}
#'   }
#'
#' @details
#' Perplexity measures the effective number of components receiving
#' meaningful weight:
#' \deqn{\text{Perplexity} = \exp\left(-\sum_i W_i \log W_i\right)}
#'
#' A perplexity of 1.0 means all weight is on one component (maximally
#' sharp). A perplexity equal to the number of components means uniform
#' weighting (no signal extraction). The default target of 2.0 (for
#' 3-component PSRI) balances signal extraction with robustness.
#'
#' The calibration evaluates perplexity across all provided profiles and
#' finds the temperature that minimizes the deviation from the target
#' mean perplexity using bisection search.
#'
#' @section Recommended workflow:
#' \enumerate{
#'   \item Compute PSRI components for a representative subset of your
#'     data (e.g., control and key treatments).
#'   \item Pass these component vectors to \code{calibrate_temperature()}.
#'   \item Use the returned \code{optimal_T} in subsequent calls to
#'     \code{\link{compute_psri_sm}}.
#' }
#'
#' @examples
#' # Calibrate using representative corn and barley profiles
#' profiles <- list(
#'   corn_control   = c(0.80, 0.90, 0.60),
#'   corn_treated   = c(0.50, 0.40, 0.55),
#'   barley_control = c(0.35, 0.30, 0.45),
#'   barley_treated = c(0.20, 0.15, 0.50)
#' )
#'
#' cal <- calibrate_temperature(profiles)
#' cal$optimal_T
#' cal$mean_perplexity
#'
#' # Use the calibrated temperature
#' compute_psri_sm(
#'   germination_counts = c(5, 15, 20),
#'   timepoints = c(3, 5, 7),
#'   total_seeds = 25,
#'   temperature = cal$optimal_T
#' )
#'
#' @seealso \code{\link{softmax_weights}}, \code{\link{compute_psri_sm}}
#'
#' @export
calibrate_temperature <- function(component_profiles,
                                  target_perplexity = 2.0,
                                  T_range = c(0.01, 2.0),
                                  tolerance = 0.05) {

  # --- Input validation ---
  if (!is.list(component_profiles) || length(component_profiles) == 0) {
    stop("component_profiles must be a non-empty list of numeric vectors.")
  }
  for (i in seq_along(component_profiles)) {
    p <- component_profiles[[i]]
    if (!is.numeric(p) || length(p) < 2) {
      stop(sprintf("Profile %d must be a numeric vector with at least 2 elements.", i))
    }
  }
  if (!is.numeric(target_perplexity) || length(target_perplexity) != 1 ||
      target_perplexity <= 0) {
    stop("target_perplexity must be a single positive number.")
  }
  if (!is.numeric(T_range) || length(T_range) != 2 ||
      T_range[1] >= T_range[2] || T_range[1] <= 0) {
    stop("T_range must be c(lower, upper) with 0 < lower < upper.")
  }

  # --- Helper: perplexity of softmax weights ---
  perplexity <- function(scores, t_val) {
    w <- softmax_weights(scores, temperature = t_val)
    h <- -sum(w * log(w + 1e-15))
    exp(h)
  }

  # --- Helper: mean perplexity across profiles ---
  mean_perplexity <- function(t_val) {
    perps <- vapply(component_profiles, function(p) {
      perplexity(p, t_val)
    }, numeric(1))
    mean(perps)
  }

  # --- Bisection search ---
  lo <- T_range[1]
  hi <- T_range[2]

  # Perplexity increases with T (more uniform at higher T)
  # We want to find T where mean perplexity = target
  perp_lo <- mean_perplexity(lo)
  perp_hi <- mean_perplexity(hi)

  # Check feasibility
  if (target_perplexity < perp_lo) {
    warning(sprintf(
      "Target perplexity %.2f is below minimum achievable (%.2f at T=%.3f). Using T_range lower bound.",
      target_perplexity, perp_lo, lo
    ))
    optimal_T <- lo
  } else if (target_perplexity > perp_hi) {
    warning(sprintf(
      "Target perplexity %.2f exceeds maximum achievable (%.2f at T=%.3f). Using T_range upper bound.",
      target_perplexity, perp_hi, hi
    ))
    optimal_T <- hi
  } else {
    # Bisect
    for (iter in seq_len(100)) {
      mid <- (lo + hi) / 2
      perp_mid <- mean_perplexity(mid)
      if (abs(perp_mid - target_perplexity) < tolerance) {
        break
      }
      if (perp_mid < target_perplexity) {
        lo <- mid
      } else {
        hi <- mid
      }
    }
    optimal_T <- (lo + hi) / 2
  }

  # --- Compute final perplexities ---
  final_perps <- vapply(component_profiles, function(p) {
    perplexity(p, optimal_T)
  }, numeric(1))

  if (!is.null(names(component_profiles))) {
    names(final_perps) <- names(component_profiles)
  }

  list(
    optimal_T            = round(optimal_T, 4),
    mean_perplexity      = mean(final_perps),
    profile_perplexities = final_perps,
    target_perplexity    = target_perplexity,
    search_range         = T_range
  )
}
