% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obn_oslp.R
\name{ob_numerical_oslp}
\alias{ob_numerical_oslp}
\title{Optimal Binning for Numerical Variables using Optimal Supervised Learning Partitioning}
\usage{
ob_numerical_oslp(
  feature,
  target,
  min_bins = 3,
  max_bins = 5,
  bin_cutoff = 0.05,
  max_n_prebins = 20,
  convergence_threshold = 1e-06,
  max_iterations = 1000,
  laplace_smoothing = 0.5
)
}
\arguments{
\item{feature}{Numeric vector of feature values. Missing values (NA) and infinite
values are \strong{not permitted} and will trigger an error.}

\item{target}{Integer or numeric vector of binary target values (must contain only
0 and 1). Must have the same length as \code{feature}. Unlike other binning
methods, OSLP internally uses \code{double} for target, allowing implicit
conversion from integer.}

\item{min_bins}{Minimum number of bins (default: 3). Must be at least 2.}

\item{max_bins}{Maximum number of bins (default: 5). Must be greater than or
equal to \code{min_bins}.}

\item{bin_cutoff}{Minimum fraction of total observations per bin (default: 0.05).
Must be in (0, 1).}

\item{max_n_prebins}{Maximum number of pre-bins (default: 20). Must be at least
equal to \code{min_bins}.}

\item{convergence_threshold}{Convergence threshold for IV change (default: 1e-6).}

\item{max_iterations}{Maximum iterations (default: 1000).}

\item{laplace_smoothing}{Laplace smoothing parameter (default: 0.5). Must be
non-negative.}
}
\value{
A list containing:
\describe{
  \item{id}{Integer bin identifiers (1-based).}
  \item{bin}{Character bin intervals \code{"[lower;upper)"}.}
  \item{woe}{Numeric WoE values (guaranteed monotonic).}
  \item{iv}{Numeric IV contributions per bin.}
  \item{count}{Integer total observations per bin.}
  \item{count_pos}{Integer positive class counts.}
  \item{count_neg}{Integer negative class counts.}
  \item{event_rate}{Numeric event rates.}
  \item{cutpoints}{Numeric bin boundaries (excluding ±Inf).}
  \item{total_iv}{Total Information Value.}
  \item{converged}{Logical convergence flag.}
  \item{iterations}{Integer iteration count.}
}
}
\description{
Implements a greedy binning algorithm with quantile-based pre-binning and
monotonicity enforcement. \strong{Important Note}: Despite "Optimal Supervised
Learning Partitioning" and "LP" in the name, the algorithm uses \strong{greedy
heuristics} without formal Linear Programming or convex optimization. The method
is functionally equivalent to \code{\link{ob_numerical_mrblp}} with minor
differences in pre-binning strategy and bin reduction criteria.

Users seeking true optimization-based binning should consider Mixed-Integer
Programming (MIP) implementations (e.g., via \code{ompr} or \code{lpSolve}
packages), though these scale poorly beyond N > 10,000 observations.
}
\details{
\strong{Algorithm Overview}

OSLP executes in five phases:

\strong{Phase 1: Quantile-Based Pre-binning}

Unlike equal-frequency methods that ensure balanced bin sizes, OSLP places
cutpoints at quantiles of \strong{unique feature values}:

\deqn{\text{edge}_i = \text{unique\_vals}\left[\left\lfloor p_i \times (n_{\text{unique}} - 1) \right\rfloor\right]}

where \eqn{p_i = i / \text{max\_n\_prebins}}.

\strong{Critique}: If unique values are clustered (e.g., many observations at
specific values), bins may have vastly different sizes, violating the equal-frequency
principle that ensures statistical stability.

\strong{Phase 2: Rare Bin Merging}

Bins with \eqn{n_i / N < \text{bin\_cutoff}} are merged. The merge direction
minimizes IV loss:

\deqn{\Delta \text{IV} = \text{IV}_i + \text{IV}_{i+d} - \text{IV}_{\text{merged}}}

where \eqn{d \in \{-1, +1\}} (left or right neighbor).

\strong{Phase 3: Initial WoE/IV Calculation}

Standard WoE with Laplace smoothing:

\deqn{\text{WoE}_i = \ln\left(\frac{n_i^{+} + \alpha}{n^{+} + k\alpha} \bigg/ \frac{n_i^{-} + \alpha}{n^{-} + k\alpha}\right)}

\strong{Phase 4: Monotonicity Enforcement}

Direction determined via majority vote (identical to MRBLP):

\deqn{\text{increasing} = \begin{cases} \text{TRUE} & \text{if } \sum_i \mathbb{1}_{\{\text{WoE}_i > \text{WoE}_{i-1}\}} \ge \sum_i \mathbb{1}_{\{\text{WoE}_i < \text{WoE}_{i-1}\}} \\ \text{FALSE} & \text{otherwise} \end{cases}}

Violations are merged iteratively.

\strong{Phase 5: Bin Count Reduction}

If \eqn{k > \text{max\_bins}}, merge bins with the \strong{smallest combined IV}:

\deqn{\text{merge\_idx} = \arg\min_{i=0}^{k-2} \left( \text{IV}_i + \text{IV}_{i+1} \right)}

\strong{Rationale}: Assumes bins with low total IV contribute least to predictive
power. However, this ignores the interaction between bins; a low-IV bin may be
essential for monotonicity or preventing gaps.

\strong{Theoretical Foundations}

Despite the name "Optimal Supervised Learning Partitioning", the algorithm lacks:
\itemize{
  \item \strong{Global optimality guarantees}: Greedy merging is myopic
  \item \strong{Formal loss function}: No explicit objective being minimized
  \item \strong{LP formulation}: No constraint matrix, simplex solver, or dual variables
}

A true optimal partitioning approach would formulate the problem as:

\deqn{\min_{\mathbf{z}, \mathbf{b}} \left\{ -\sum_{i=1}^{k} \text{IV}_i(\mathbf{b}) + \lambda k \right\}}

subject to:
\deqn{\sum_{j=1}^{k} z_{ij} = 1 \quad \forall i \in \{1, \ldots, N\}}
\deqn{\text{WoE}_j \le \text{WoE}_{j+1} \quad \forall j}
\deqn{z_{ij} \in \{0, 1\}, \quad b_j \in \mathbb{R}}

where \eqn{z_{ij}} indicates observation \eqn{i} assigned to bin \eqn{j}, and
\eqn{\lambda} is a complexity penalty. This requires MILP solvers (CPLEX, Gurobi)
and is intractable for \eqn{N > 10^4}.

\strong{Comparison with Related Methods}

\tabular{lllll}{
  \strong{Method} \tab \strong{Pre-binning} \tab \strong{Direction} \tab \strong{Merge (max_bins)} \tab \strong{Target Type} \cr
  OSLP \tab Quantile (unique vals) \tab Majority vote \tab Min (IV(i) + IV(i+1)) \tab double \cr
  MRBLP \tab Equal-frequency \tab Majority vote \tab Min |IV(i) - IV(i+1)| \tab int \cr
  MOB \tab Equal-frequency \tab First two bins \tab Min IV loss \tab int \cr
  MBLP \tab Quantile (data) \tab Correlation \tab Min IV loss \tab int \cr
}

\strong{When to Use OSLP}

\itemize{
  \item \strong{Use OSLP}: Never. Use MBLP or MOB instead for better pre-binning
    and merge strategies.
  \item \strong{Use MBLP}: For robust direction detection via correlation.
  \item \strong{Use MDLP}: For information-theoretic stopping criteria.
  \item \strong{Use True LP}: For small datasets (N < 1000) where global optimality
    is critical and computational cost is acceptable.
}
}
\examples{
\donttest{
set.seed(123)
n <- 5000
feature <- c(
  rnorm(2000, 600, 50),
  rnorm(2000, 680, 40),
  rnorm(1000, 740, 30)
)
target <- c(
  rbinom(2000, 1, 0.25),
  rbinom(2000, 1, 0.10),
  rbinom(1000, 1, 0.03)
)

result <- ob_numerical_oslp(
  feature = feature,
  target = target,
  min_bins = 3,
  max_bins = 5
)

print(result$woe)
print(result$total_iv)

# Compare with MRBLP (should be nearly identical)
result_mrblp <- ob_numerical_mrblp(
  feature = feature,
  target = target,
  min_bins = 3,
  max_bins = 5
)

data.frame(
  Method = c("OSLP", "MRBLP"),
  Total_IV = c(result$total_iv, result_mrblp$total_iv),
  N_Bins = c(length(result$woe), length(result_mrblp$woe))
)
}

}
\references{
\itemize{
  \item Mironchyk, P., & Tchistiakov, V. (2017). "Monotone optimal binning algorithm
    for credit risk modeling". \emph{Frontiers in Applied Mathematics and Statistics}, 3, 2.
  \item Zeng, G. (2014). "A Necessary Condition for a Good Binning Algorithm in
    Credit Scoring". \emph{Applied Mathematical Sciences}, 8(65), 3229-3242.
  \item Fayyad, U. M., & Irani, K. B. (1993). "Multi-Interval Discretization of
    Continuous-Valued Attributes". \emph{IJCAI}, pp. 1022-1027.
  \item Good, I. J. (1952). "Rational Decisions". \emph{Journal of the Royal
    Statistical Society B}, 14(1), 107-114.
  \item Siddiqi, N. (2006). \emph{Credit Risk Scorecards}. Wiley.
}
}
\seealso{
\code{\link{ob_numerical_mrblp}} for nearly identical algorithm with better pre-binning,
\code{\link{ob_numerical_mblp}} for correlation-based direction detection,
\code{\link{ob_numerical_mdlp}} for information-theoretic optimality.
}
\author{
Lopes, J. E.
}
