#
#  repeated : A Library of Repeated Measurements Models
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     kalcount(response, times=NULL, origin=0, intensity="exponential",
#	depend="independence", update="Markov", mu=NULL, shape=NULL,
#	density=F, ccov=NULL, tvcov=NULL, pinitial=1, pdepend=NULL,
#	pshape=NULL, preg=NULL, pbirth=NULL, ptvc=NULL, pintercept=NULL,
#	print.level=0, ndigit=10, gradtol=0.00001, steptol=0.00001,
#	fscale=1, iterlim=100, typsiz=abs(p), stepmax=10*sqrt(p%*%p))
#
#  DESCRIPTION
#
#    A function to fit various distributions inserted into a Pareto
#  distribution with serial dependence or gamma frailties using
#  Kalman-type update for longitudinal count data.

kalcount <- function(response, times=NULL, origin=0, intensity="exponential",
	depend="independence", update="Markov", mu=NULL, shape=NULL,
	density=F, ccov=NULL, tvcov=NULL, pinitial=1, pdepend=NULL,
	pshape=NULL, preg=NULL, pbirth=NULL, ptvc=NULL, pintercept=NULL,
	print.level=0, ndigit=10, gradtol=0.00001, steptol=0.00001,
	fscale=1, iterlim=100, typsiz=abs(p), stepmax=10*sqrt(p%*%p)){
kcountb <- function(p){
	if(rf)b <- mu(p)
	if(sf)v <- shape(p[nps1:np])
	z <- .C("kcountb",
		p=as.double(p),
		y=as.double(zna$response$times),
		origin=as.double(origin),
		c=as.integer(zna$response$y),
		x=as.double(zna$ccov$ccov),
		nind=as.integer(nind),
		nobs=as.integer(zna$response$nobs),
		nbs=as.integer(length(zna$response$y)),
		nccov=as.integer(nccov),
		model=as.integer(mdl),
		density=as.integer(density),
		dep=as.integer(dep),
		birth=as.integer(birth),
		tvc=as.integer(tvc),
		tvcov=as.double(zna$tvcov$tvcov),
		fit=as.integer(0),
		pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
		rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
		like=double(1))
	z$like}
countfb <- function(p){
	if(rf)b <- mu(p)
	if(sf)v <- shape(p[nps1:np])
	z <- .C("countfb",
		p=as.double(p),
		y=as.double(zna$response$times),
		c=as.integer(zna$response$y),
		x=as.double(zna$ccov$ccov),
		nind=as.integer(nind),
		nobs=as.integer(zna$response$nobs),
		nbs=as.integer(length(zna$response$y)),
		nccov=as.integer(nccov),
		model=as.integer(mdl),
		density=as.integer(density),
		tvc=as.integer(tvc),
		tvcov=as.double(zna$tvcov$tvcov),
		fit=as.integer(0),
		pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
		rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
		like=double(1))
	z$like}
call <- sys.call()
tmp <- c("exponential", "Weibull","gamma","gen logistic","log normal",
	"log logistic","log Cauchy","log Laplace")
mdl <- match(intensity <- match.arg(intensity,tmp),tmp)
depend <- match.arg(depend,c("independence","serial","frailty"))
tmp <- c("Markov","serial","event","cumulated","count","kalman","time")
dep <- match(update <- match.arg(update,tmp),tmp)
v <- b <- NULL
rf <- !missing(mu)
sf <- !missing(shape)
if(rf&&!is.function(mu))stop("mu must be a function")
if(sf&&!is.function(shape))stop("shape must be a function")
if(origin<0)stop("Origin must be positive")
npreg <- length(preg)
birth <- !missing(pbirth)
tvc <- length(ptvc)
if(rf&&birth)stop("Birth models cannot be fitted with a mean function")
if(intensity=="exponential"){
	sf <- F
	pshape <- NULL}
else {
	if(missing(pshape))
		stop("Initial value of the shape parameter must be supplied")
	if(!sf){
		if(pshape<=0)stop("shape must be positive")
		pshape <- log(pshape)}}
if(intensity=="gen logistic"){
	if(is.null(pintercept))stop("Initial value of the intercept parameter must be supplied")}
else pintercept <- NULL
if(pinitial<=0)stop("Estimate of initial parameter must greater than 0")
else pinitial <- log(pinitial)
if(depend=="independence"){
	pdepend <- NULL
	dep <- 0}
else if(depend=="serial"){
	if(update=="time")stop("time update can only be used with frailty")
	if(missing(pdepend))
		stop("An estimate of the dependence parameter must be supplied")
	else if(pdepend<=0|pdepend>=1)
		stop("Dependence parameter must be between 0 and 1")
	else pdepend <- log(pdepend/(1-pdepend))}
else if(depend=="frailty"){
	if(update=="time")dep <- 1
	else {
		dep <- 0
		update <- "no"}
	if(!missing(pdepend))pdepend <- NULL}
if(missing(ccov)){
	nccov <- 0
	zc <- NULL}
else {
	if(is.vector(ccov,mode="double")||(is.matrix(ccov)&&is.null(colnames(ccov))))
		ccname <- paste(deparse(substitute(ccov)))
	else ccname <- NULL
	zc <- tcctomat(ccov,names=ccname)
	nccov <- ncol(zc$ccov)}
if(rf&&npreg>0)nccov <- npreg-1
if(!rf&&1+nccov!=npreg)
	stop(paste(1+nccov,"regression estimates must be supplied"))
if(inherits(response,"response"))zr <- response
else zr <- restovec(response,times)
if(!missing(tvcov)){
	if(inherits(tvcov,"tvcov"))zt <- tvcov
	else {
		if(is.matrix(tvcov)||is.data.frame(tvcov)||
			(is.list(tvcov)&&is.null(colnames(tvcov[[1]]))))
			tvcname <- paste(deparse(substitute(tvcov)))
		else tvcname <- NULL
		zt <- tvctomat(tvcov, names=tvcname)}
	ttvc <- ncol(zt$tvcov)}
else {
	ttvc <- 0
	zt <- NULL}
zna <- rmna(response=zr, tvcov=zt, ccov=zc)
rm(zr,zt,zc)
nind <- length(zna$response$nobs)
if(ttvc>0&&tvc!=ttvc)stop(paste(ttvc,"initial estimates of coefficients for time-varying covariates must be supplied"))
if(rf){
	if(tvc>0&&nccov>0)stop("With a mean function, initial estimates must be supplied either in preg or in ptvc")
	if(tvc>0){
		if(length(mu(ptvc))!=length(zna$response$y))stop("The mu function must provide an estimate for each observation")
		tvc <- tvc-1}
	else if(length(mu(preg))==1){
		if(nccov==0)mu <- function(p) rep(p[1],length(zna$response$y))
		else stop("Number of estimates does not correspond to mu function")}
	else if(length(mu(preg))!=nind)stop("The mu function must provide an estimate for each individual")}
if(sf&&length(shape(pshape))!=length(zna$response$y))stop("The shape function must provide an estimate for each observation")
nps <- length(pshape)
np <- 1+nccov+tvc+1+(depend=="serial")+birth+nps+!is.null(pintercept)
nps1 <- np-nps-!is.null(pintercept)+1
p <- c(preg,pbirth,ptvc,pinitial,pdepend,pshape,pintercept)
if(depend=="frailty")count <- countfb
else count <- kcountb
if(fscale==1)fscale <- count(p)
z0 <- nlm(count, p=p, hessian=T, print.level=print.level,
	typsiz=typsiz, ndigit=ndigit, gradtol=gradtol, stepmax=stepmax,
	steptol=steptol, iterlim=iterlim, fscale=fscale)
if(any(is.na(z0$hessian)))a <- 0
else a <- qr(z0$hessian)$rank
if(a==np)cov <- solve(z0$hessian)
else cov <- matrix(NA,ncol=np,nrow=np)
se <- sqrt(diag(cov))
corr <- cov/(se%o%se)
dimnames(corr) <- list(1:np,1:np)
z <- if(depend=="frailty"){
	if(rf)b <- mu(z0$estimate)
	if(sf)v <- shape(z0$estimate[nps1:np])
	.C("countfb",
        	p=as.double(z0$estimate),
        	y=as.double(zna$response$times),
        	c=as.integer(zna$response$y),
        	x=as.double(zna$ccov$ccov),
        	nind=as.integer(nind),
        	nobs=as.integer(zna$response$nobs),
        	nbs=as.integer(length(zna$response$y)),
        	nccov=as.integer(nccov),
        	model=as.integer(mdl),
        	density=as.integer(density),
        	tvc=as.integer(tvc),
        	tvcov=zna$tvcov$tvcov,
        	fit=as.integer(1),
        	pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
        	rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
        	like=double(1))}
else {
	if(rf)b <- mu(z0$estimate)
	if(sf)v <- shape(z0$estimate[nps1:np])
	.C("kcountb",
        	p=as.double(z0$estimate),
        	y=as.double(zna$response$times),
        	origin=as.double(origin),
        	c=as.integer(zna$response$y),
        	x=as.double(zna$ccov$ccov),
        	nind=as.integer(nind),
        	nobs=as.integer(zna$response$nobs),
        	nbs=as.integer(length(zna$response$y)),
        	nccov=as.integer(nccov),
        	model=as.integer(mdl),
        	density=as.integer(density),
        	dep=as.integer(dep),
        	birth=as.integer(birth),
        	tvc=as.integer(tvc),
        	tvcov=zna$tvcov$tvcov,
        	fit=as.integer(1),
        	pred=double(length(zna$response$y)),
		rpred=double(length(zna$response$y)),
		rf=as.integer(rf),
		bb=as.double(b),
		sf=as.integer(sf),
		vv=as.double(v),
		like=double(1))}
z <- list(
	call=call,
	intensity=intensity,
	mdl=mdl,
	mu=mu,
	npr=1+nccov+tvc+birth,
	shape=shape,
	nps=np-nps,
	density=density,
	depend=depend,
	update=update,
	birth=birth,
	response=zna$response,
	pred=z$pred,
	rpred=z$rpred,
	ccov=zna$ccov,
	tvcov=zna$tvcov,
	maxlike=z0$minimum,
	aic=z0$minimum+np,
	df=length(zna$response$y)-np,
	npt=np,
	coefficients=z0$estimate,
	se=se,
	cov=cov,
	corr=corr,
	grad=z0$gradient,
	iterations=z0$iterations,
	code=z0$code)
class(z) <- c("kalcount","recursive")
return(z)}

coefficients.kalcount <- function(z) z$coefficients
deviance.kalcount <- function(z) 2*z$maxlike
fitted.kalcount <- function(z, recursive=TRUE)
	if(recursive) z$rpred else z$pred
residuals.kalcount <- function(z, type = "response", recursive=TRUE){
	if(type=="response") z$response$y-z$rpred
	else (z$response$y-z$rpred)/sqrt(z$rpred)}

print.kalcount <- function(z, digits = max(3, .Options$digits - 3)) {
	if(!is.null(z$ccov))nccov <- ncol(z$ccov$ccov)
	else nccov <- 0
	expm <- z$intensity!="exponential"&&!is.function(z$shape)
	glm <- z$intensity=="gen logistic"
	nps <- if(is.function(z$shape)) z$nps else z$npt
	deppar <- (z$depend=="serial"||z$depend=="Markov")
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	if(z$code>2)cat("Warning: no convergence - error",z$code,"\n\n")
	cat("Number of subjects    ",length(z$response$nobs),"\n")
	cat("Number of observations",length(z$response$y),"\n")
	if(z$density)cat(z$intensity," density",sep="")
	else cat(z$intensity," intensity",sep="")
	if(z$depend=="independence")cat(" with independence\n")
	else if(z$depend=="frailty")
		cat(" with",z$depend,"dependence and",z$update,"weight\n")
	else cat(" with ",z$update," update\n",sep="")
	cat("\n-Log likelihood   ",z$maxlike,"\n")
	cat("Degrees of freedom",z$df,"\n")
	cat("AIC               ",z$aic,"\n")
	cat("Iterations        ",z$iterations,"\n\n")
	cat("Location parameters\n")
	if(is.function(z$mu)){
		t <- deparse(z$mu)
		cat("Function:",t[2:length(t)],sep="\n")}
	else if(is.language(z$ccov$linear))
		cat("Formula: ",deparse(z$ccov$linear),"\n")
	coef.table <- cbind(z$coef[1:z$npr],z$se[1:z$npr])
	if(!is.function(z$mu)){
		cname <- "(Intercept)"
		if(nccov)cname <- c(cname,colnames(z$ccov$ccov))
		if(z$birth)cname <- c(cname,"birth")
		if(!is.null(z$tvcov))cname <- c(cname,colnames(z$tvcov$tvcov))}
	else {
		cname <- NULL
		for(i in 1:nrow(coef.table))cname <- c(cname,paste("p",i,sep=""))}
	dimnames(coef.table) <- list(cname, c("estimate","se"))
	print.default(coef.table, digits=digits, print.gap=2)
	if(is.function(z$shape))cat("\nDependence parameters\n")
	else cat("\nNonlinear parameters\n")
	coef <- exp(z$coef[(nps-deppar-expm-glm):nps])
	cname <- "initial"
	if(deppar){
		coef[2] <- coef[2]/(1+coef[2])
		cname <- c(cname,"depend")}
	if(glm){
		cname <- c(cname,"asymptote","intercept")
		coef[length(coef)-1] <- 1/coef[length(coef)-1]
		coef[length(coef)] <- NA}
	else if(expm)cname <- c(cname,"shape")
	coef.table <- cbind(z$coef[(nps-deppar-expm-glm):nps],z$se[(nps-deppar-expm-glm):nps],coef)
	dimnames(coef.table) <- list(cname, c("estimate","se","parameter"))
	print.default(coef.table, digits=digits, print.gap=2)
	if(is.function(z$shape)){
		cat("\nShape parameters\n")
		t <- deparse(z$shape)
		cat("Function:",t[2:length(t)],sep="\n")
		coef.table <- cbind(z$coef[(z$nps+1):z$npt],
			z$se[(z$nps+1):z$npt])
		cname <- NULL
		for(i in 1:nrow(coef.table))
			cname <- c(cname,paste("p",i,sep=""))
		dimnames(coef.table) <- list(cname, c("estimate","se"))
		print.default(coef.table, digits=digits, print.gap=2)}
	cat("\nCorrelation matrix\n")
	print.default(z$corr, digits=digits)}
