\name{elliptic}
\title{Multivariate elliptically-contoured repeated measurements
models for linear and nonlinear changes over time in the presence of
time-varying covariates and with AR(1) and two levels of variance
components}
\usage{
elliptic(response, model="linear", times=NULL, dose=NULL, ccov=NULL,
	tvcov=NULL, nest=NULL, torder=0, interaction=NULL,
	transform="identity", link="identity", autocorr="exponential",
	pell=NULL, preg=rep(1,4), pvar=1, varfn=NULL, pre=NULL,
	par=NULL, delta=NULL, print.level=0, gradtol=0.00001,
	typsiz=abs(theta), stepmax=10*sqrt(theta\%*\%theta),
	steptol=0.00001, iterlim=100, ndigit=10, fscale=1)
}
\alias{elliptic}
\alias{coefficients.elliptic}
\alias{deviance.elliptic}
\alias{print.elliptic}
\alias{residuals.elliptic}
\arguments{
\item{response}{A list of two or three column matrices with response
values, times, and possibly nesting categories, for each individual,
one matrix or dataframe of response values, or an object of class,
response (created by \code{\link{restovec}}).}
\item{model}{The model to be fitted for the location. Builtin choices
are (1) \code{linear} for linear models with time-varying covariate; if
\code{torder} > 0, a polynomial in time is automatically fitted; (2)
\code{logistic} for a four-parameter logistic growth curve; (3)
\code{pkpd} for a first-order one-compartment pharmacokinetic model.
Otherwise, set this to a function of the parameters that describes the
location, returning a vector the same length as the number of
observations, in which case ccov and tvcov cannot be used.}
\item{times}{When \code{response} is a matrix, a vector of possibly
unequally spaced times when they are the same for all individuals or a
matrix of times. Not necessary if equally spaced.}
\item{dose}{A vector of dose levels for the \code{pkpd model}.}
\item{ccov}{A vector or matrix containing time-constant baseline
covariates with one line per individual, a model formula using
vectors of the same size, or an object of class, tccov (created by
\code{\link{tcctomat}}). For the \code{pkpd} and \code{logistic}
models, all variables must be binary (or factor variables) as
different values of all parameters are calculated for all combinations
of these variables (except for the logistic model when a time-varying
covariate is present). Cannot be used when model is a function.}
\item{tvcov}{A list of vectors or matrices with time-varying
covariates for each individual (one column per variable), a matrix or
dataframe of such covariate values (if only one covariate), or an
object of class, tvcov (created by \code{\link{tvctomat}}). If times
are not the same as for responses, the list can be created with
\code{\link{gettvc}}. Only one time-varying covariate is allowed
except for the \code{linear model}; if more are required, set
\code{model} equal to the appropriate mean function. Cannot be used
when model is a function.}
\item{nest}{When \code{response} is a matrix, a vector of length equal
to the number of responses per individual indicating which responses
belong to which nesting category. Categoriess must be consecutive
increasing integers. This option should always be specified if nesting
is present.}
\item{torder}{When the \code{linear model} is chosen, order of the
polynomial in time to be fitted.}
\item{interaction}{Vector of length equal to the number of
time-constant covariates, giving the levels of interactions between
them and the polynomial in time in the \code{linear model}.}
\item{transform}{Transformation of the response variable: \code{identity},
\code{exp}, \code{square}, \code{sqrt}, or \code{log}.}
\item{link}{Link function for the location: \code{identity}, \code{exp},
\code{square}, \code{sqrt}, or \code{log}. For the \code{linear model},
if not the \code{identity}, initial estimates of the regression
parameters must be supplied (intercept, polynomial in time,
time-constant covariates, time-varying covariates, in that order).}
\item{autocorr}{The form of the autocorrelation function:
\code{exponential} is the usual rho^|t_i-t_j|; \code{gaussian} is
rho^((t_i-t_j)^2); \code{cauchy} is 1/(1+rho(t_i-t_j)^2);
\code{spherical} is ((|t_i-t_j|rho)^3-3|t_i-t_j|rho+2)/2 for
|t_i-t_j|<=1/rho and zero otherwise; \code{IOU} is the integrated
Ornstein-Uhlenbeck process, (2rho min(t_i,t_j)+exp(-rho t_i)
+exp(-rho t_j)-1 -exp(rho|ti-t_j|))/2rho^3.}
\item{pell}{Initial estimate of the power parameter of the
multivariate elliptically-contoured distribution. If missing, the
multivariate normal distribution is used.}
\item{preg}{Initial parameter estimates for the regression model.
Only required for \code{linear model} if the \code{link} is not the
\code{identity} or a variance function is fitted.}
\item{pvar}{Initial parameter estimate for the variance. If more than
one value is provided, the log variance depends on a polynomial in
time. With the \code{pkpd model}, if four values are supplied, a
nonlinear regression for the variance is fitted.}
\item{varfn}{The builtin variance function has the variance
proportional to a function of the location:  pvar*v(mu) = \code{identity}
or \code{square}. If pvar contains two initial values, an additive
constant is included: pvar(1)+pvar(2)*v(mu). Otherwise, a function for
the log variance can be supplied, returning a vector the same length
as the number of observations.}
\item{pre}{Zero, one or two parameter estimates for the variance
components, depending on the number of levels of nesting.}
\item{par}{If supplied, an initial estimate for the autocorrelation
parameter.}
\item{delta}{Scalar or vector giving the unit of measurement
for each response value, set to unity by default. For example, if a
response is measured to two decimals, delta=0.01.}
\item{others}{Arguments controlling \code{\link{nlm}}.}
}
\description{
\code{elliptic} fits a special case of the multivariate
elliptically-contoured distribution, called the multivariate power
exponential distribution. It includes the multivariate normal
(power=1), the multivariate Laplace (power=0.5), and the multivariate
uniform (power -> infinity) distributions as special cases.

It is designed to fit linear and nonlinear models with time-varying
covariates observed at arbitrary time points. A continuous-time AR(1)
and zero, one, or two levels of nesting can be handled. 
}
\value{
A list of class \code{elliptic} is returned.
}
\seealso{
\code{\link{carma}}, \code{\link{gar}}, \code{\link{gettvc}},
\code{\link{glmm}}, \code{\link{gnlmm}}, \code{\link{gnlr}},
\code{\link{kalseries}}, \code{\link{potthoff}}, \code{\link{read.list}},
\code{\link{restovec}}, \code{\link{rmna}}, \code{\link{tcctomat}},
\code{\link{tvctomat}}.
}
\examples{
# generalized logistic regression with square-root transformation
# and square  link - data are completely unbalanced so that y is
# a list
elliptic(y, model="logistic", preg=c(5.7,4.6,1.7,-1.5), pvar=26.4,
	trans="sqrt", link="square")
# the same model with AR(1) and one component of variance
elliptic(y, model="logistic", preg=c(5.7,3.9,1.4,-1.5), pvar=4.3,
	pre=7.4, par=0.8, trans="sqrt", link="square")
# addition of a factor variable containing three levels (two dummy
# variables) - this yields three estimates of each of the four
# parameters
elliptic(y, model="logistic", preg=c(5.7,4,1.4,-1,0,1,0,0,0,0.3,0,0),
	pvar=4.3, pre=6.7, par=0.9, ccov=~treat, trans="sqrt",
	link="square")
# addition of a time-varying covariate influencing the second
# nonlinear parameter - for this model, the factor variable now
# only relates to this parameter as well
elliptic(y, model="logistic", preg=c(5.6,1.2,-1.4,0.5,1,1.3), pvar=2,
	par=0.9, pre=10.4, ccov=~treat, tvcov=dose, trans="sqrt",
	link="square")
# the same logistic model specified as a mean function of time (tt) only
mu <- function(p){
	y0 <- exp(p[1])
	yinf <- exp(p[2])
	yinf*(1+((yinf/y0)^p[4]-1)*exp(-yinf^p[4]*
		exp(p[3])*tt))^(-1/p[4])}
elliptic(y, model=mu, preg=c(5.7,4.6,1.7,-1.5), pvar=26.4, torder=1,
	trans="sqrt", link="square")
# first-order one-compartment pharmacokinetic model with log normal
# distribution and exp link - data are balanced so that z is a
# matrix but times are unequally spaced, given in times
elliptic(z, model="pkpd", preg=c(-1,0.8,-2.5), pvar=1, times=times,
	dose=dose, trans="log", link="exp")
# same model with supplied mean function, where d is a dose vector
mf <- function(p)
	exp(p[1]-p[3])*(d/(exp(p[1])-exp(p[2]))*
		(exp(-exp(p[2])*tt)-exp(-exp(p[1])*tt)))
elliptic(z, model=mf, preg=c(-1,0.8,-2.5), pvar=1, times=times,
	trans="log", link="exp")
# add nonconstant variance, two nested random components and AR(1)
# dose becomes a time-varying covariate (tvcov)
elliptic(z, model="pkpd", preg=c(-1.1,1.5,-6.3), times=times,
	pvar=c(-0.6,-1.1,-0.3,-0.5), pre=c(0.1,0.3), par=0.3,
	trans="log", link="exp", tvcov=tvcov,
	nest=c(rep(1,12),rep(2,12),rep(3,12)))
# same model with supplied mean and variance functions
mf <- function(p)
	exp(p[1]-p[3])*(d/(exp(p[1])-exp(p[2]))*
		(exp(-exp(p[2])*tt)-exp(-exp(p[1])*tt)))
vf <- function(p)
	p[1]-p[3]+p[4]*log(d/(exp(p[1])-exp(p[2]))*
		(exp(-exp(p[2])*tt)-exp(-exp(p[1])*tt)))
elliptic(z, model=mf, preg=c(-1.1,1.5,-6.3), times=times,
	pvar=c(-0.6,-1.1,-0.3,-0.5), pre=c(0.1,0.3), par=0.3,
	trans="log", link="exp", varfn=vf,
	nest=c(rep(1,12),rep(2,12),rep(3,12)))
}
