#
#  gnlm : A Library of Special Functions for Nonlinear Regression
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     nordr(y, dist="proportional", mu, linear=NULL, pmu, 
#      pintercept, wt=NULL, print.level=0, ndigit=10, gradtol=0.00001,
#      steptol=0.00001, fscale=1, iterlim=100, typsiz=abs(p),
#      stepmax=10*sqrt(p%*%p))
#
#  DESCRIPTION
#
#    A function to fit nonlinear regression models for ordinal responses.

nordr <- function(y, dist="proportional", mu, linear=NULL, pmu, 
      pintercept, wt=NULL, print.level=0, ndigit=10, gradtol=0.00001,
      steptol=0.00001, fscale=1, iterlim=100, typsiz=abs(p),
      stepmax=10*sqrt(p%*%p)){
lf <- function(p){
   g <- exp(mu1(p[1:npl])+block%*%p[npl1:np])
   g <- g/(1+g)
   if(mdl==1){
	g <- c(g,ext)
	g <- g[1:nlen]/g[nrows1:nlenr]
	g <- ifelse(g>=1,0.99,g)}
   -sum(pwt*(resp*log(g)+(1-resp)*log(1-g)))
}
lf3 <- function(p){
    mu <- mu1(p[1:npl])
    g <- exp(mu*(y-1)+resp%*%p[npl1:np])/
      exp(mu%o%(0:my)+matrix(rep(cumsum(c(0,0,p[npl1:np])),nrows),ncol=my+1,byrow=T))%*%ext
    -sum(pwt*log(g))}
call <- sys.call()
tmp <- c("proportional odds","continuation ratio","adjacent categories")
mdl <- match(dist <- match.arg(dist,tmp),tmp)
if(min(y)!=1)stop("ordinal values must be start at 1")
else if(any(y!=trunc(y)))stop("ordinal values must be integers")
else my <- max(y)-1
nrows <- length(y)
nrows1 <- nrows+1
nlen <- my*nrows
nlenr <- nlen+nrows
nlp <- npl <- length(pmu)
npl1 <- npl+1
if(is.language(mu))linear <- mu
if(is.language(linear)){
	mt <- terms(linear)
	if(is.numeric(mt[[2]])){
		dm1 <- matrix(1)
		colnames(dm1) <- "(Intercept)"
		if(!is.function(mu)){
			mu1 <- function(p) p[1]*rep(1,nrows)
			nlp <- 1}
		else mu1 <- function(p) mu(p, p[1]*rep(1,nrows))}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm1 <- model.matrix(mt, mf)
		npt1 <- ncol(dm1)
		if(!is.function(mu)){
			mu1 <- function(p) as.vector(dm1%*%p[1:npt1])
			nlp <- npt1}
		else mu1 <- function(p) mu(p, dm1%*%p[1:npt1])}}
else if(!is.function(mu)){
	mu1 <- function(p) p[1]*rep(1,nrows)
	nlp <- 1}
else {
	mu1 <- mu
	if(length(mu1(pmu))==1)mu1 <- function(p) mu(p)*rep(1,nrows)}
if(nlp!=npl)
	stop("Number of initial estimates for mu does not correspond to model")
if(mdl==1)ext <- rep(1,nrows)
else if(mdl==3)ext <- rep(1,my+1)
if(mdl==3)resp <- NULL
else resp <- matrix(as.integer(y==1),ncol=1)
block <- NULL
pwt <- matrix(as.integer(y<3),ncol=1,nrow=nrows)
for(i in 2:my){
      resp <- cbind(resp,as.integer(y<=i))
      block <- cbind(block,as.integer(c(rep(0,nrows*(i-1)),
	    rep(1,nrows),rep(0,nrows*(my-i)))))
      pwt <- cbind(pwt,as.integer(y<i+2))}
if(mdl!=1)resp <- 1-resp
if(mdl!=3){
	resp <- as.vector(resp)
	pwt <- as.vector(pwt)}
else pwt <- rep(1,length(y))
if(!is.null(wt)){
	if(!is.vector(wt,mode="double"))stop("wt must be a vector")
	else if(length(wt)!=nrows)stop("wt must have length",nrows)
	if(mdl==3)pwt <- wt
	else pwt <- rep(wt,my)*pwt}
if(missing(pintercept)||length(pintercept)!=my-1)
	stop(paste(my-1,"initial values of intercept parameters must be supplied"))
p <- c(pmu,pintercept)
np <- length(p)
if(mdl==3)z <- nlm(lf3, p, hessian=T, print.level=print.level,
	typsiz=typsiz, ndigit=ndigit, gradtol=gradtol, stepmax=stepmax,
	steptol=steptol, iterlim=iterlim, fscale=fscale)
else z <- nlm(lf, p, hessian=T, print.level=print.level,
	typsiz=typsiz, ndigit=ndigit, gradtol=gradtol, stepmax=stepmax,
	steptol=steptol, iterlim=iterlim, fscale=fscale)
maxlike <- z$minimum
if(!is.language(linear))cname <- paste("p",1:npl,sep="")
else {
     cname <- colnames(dm1)
     if(is.function(mu))
	cname <- c(cname,paste("p",(length(cname)+1):npl,sep=""))}
a <- qr(z$hessian)
if(a$rank==np)cov <- solve(z$hessian)
else cov <- matrix(NA,ncol=np,nrow=np)
se <- sqrt(diag(cov))
corr <- cov/(se%o%se)
dimnames(corr) <- list(1:np,1:np)
z1 <- list(
   call=call,
   dist=dist,
   wt=wt,
   maxlike=maxlike,
   aic=maxlike+np,
   mu=mu,
   linear=linear,
   coefficients=z$estimate[1:npl],
   cname=cname,
   np=np,
   npl=npl1-1,
   nrows=nrows,
   intercept=z$estimate[npl1:np],
   cov=cov,
   corr=corr,
   se=se,
   iterations=z$iter,
   code=z$code)
class(z1) <- "nordr"
z1}

print.nordr <- function(z, digits = max(3, .Options$digits - 3)){
	m <- z$states
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	cat(z$dist,"model\n\n")
	if(z$code>2)cat("Warning: no convergence - error",z$code,"\n\n")
	cat("-Log likelihood   ",z$maxlike,"\n")
	cat("AIC               ",z$aic,"\n")
	cat("Iterations        ",z$iterations,"\n")
	cat("\nMean coefficients\n")
	if(is.function(z$mu)){
		t <- deparse(z$mu)
		cat("Mean function:\n",t[2:length(t)],sep="\n")
		if(is.language(z$linear))
			cat("Linear part: ",deparse(z$linear[[2]]),"\n")}
	coef.table <- cbind(z$coef,z$se[1:z$npl])
	dimnames(coef.table) <- list(z$cname,c("estimate","s.e."))
	print.default(coef.table, digits=digits, print.gap=2)
	cat("\nIntercept coefficients\n")
	coef.table <- cbind(z$intercept,z$se[(z$npl+1):z$np])
	dimnames(coef.table) <- list(paste("b",2:(z$np-z$npl+1),sep=""),
			     c("estimate","s.e."))
	print.default(coef.table, digits=digits, print.gap=2)
	cat("\nCorrelation matrix\n")
	print.default(z$corr, digits=digits)
}
