#
#  gnlm : A Library of Special Functions for Nonlinear Regression
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     fmr(y, dist="normal", pmu=NULL, pmix=NULL, pshape=NULL, mu=NULL,
#	mix=NULL, linear=NULL, censor="right",exact=F, wt=1, delta=1,
#	print.level=0, typsiz=abs(p), ndigit=10, gradtol=0.00001,
#	stepmax=10*sqrt(p%*%p), steptol=0.00001, iterlim=100, fscale=1)
#
#  DESCRIPTION
#
#    A function to fit nonlinear regression models with a variety of
# distributions and a mixture in the tail(s).

fmr <- function(y, dist="normal", pmu=NULL, pmix=NULL, pshape=NULL, mu=NULL,
	mix=NULL, linear=NULL, censor="right",exact=F, wt=1, delta=1,
	print.level=0, typsiz=abs(p), ndigit=10, gradtol=0.00001,
	stepmax=10*sqrt(p%*%p), steptol=0.00001, iterlim=100,
	fscale=1){

pinvgauss <- function(y,m,s){
	t <- y/m
	v <- sqrt(y*s)
	pnorm((t-1)/v)+exp(2/(m*s))*pnorm(-(t+1)/v)}
plaplace <- function(y){
	t <- exp(-abs(y))/2
	ifelse(y<0,t,1-t)}
plevy <- function(y, m, s)
	.C("plevy",
		as.double(y),
		as.double(m),
		as.double(s),
		as.double(1),
		len=as.integer(n),
		eps=as.double(1.0e-6),
		pts=as.integer(5),
		max=as.integer(16),
		err=integer(1),
		res=double(n))$res

call <- sys.call()
if(!missing(dist)&&!is.function(dist)){
	dist <- match.arg(dist,c("binomial","beta binomial","double binomial",
	"mult binomial","Poisson","negative binomial","double Poisson",
	"mult Poisson","gamma count","Consul","geometric","normal",
	"inverse Gauss","logistic","exponential","gamma","Weibull",
	"extreme value","Pareto","Cauchy","Student t","Laplace","Levy"))}
if(!missing(pmu))npl <- length(pmu)
else npl <- 0
if(!missing(pmix))npm <- length(pmix)
else npm <- 0
sht <- dist!="binomial"&&dist!="Poisson"&&dist!="exponential"&&dist!="geometric"
if(sht&&missing(pshape))
	stop("An estimate of the shape parameter must be given")
np <- npl+npm+sht
p <- c(pmu,pmix,pshape)
if(is.function(dist)){
	fcn <- dist
	dist <- "own"}
if(inherits(y,"response")){
	if(is.null(y$censor))y <- y$y
	else y <- cbind(y$y,y$censor)}
if(dist=="Poisson"||dist=="negative binomial"||dist=="double Poisson"||dist=="mult Poisson"||dist=="gamma count"||dist=="Consul"){
	if(!is.vector(y,mode="double"))stop("y must be a vector")
	n <- length(y)
	censor <- NULL
	cens <- ifelse(y==0,1,0)}
else {
	if(length(dim(y))!=2||ncol(y)!=2)
		stop(paste("Two column matrix required for response:",
		if(dist=="binomial"||dist=="beta binomial"||dist=="double binomial"||dist=="mult binomial")"successes and failures"
		else "times and censor indicator"))
	else {
		n <- nrow(y)
		if(dist=="binomial"||dist=="beta binomial"||dist=="double binomial"||dist=="mult binomial"){
			if(missing(censor))
				stop("Censoring must be left, right, or both")
			if(censor!="left"&&censor!="right"&&censor!="both")
				stop("Censoring must be left, right, or both")
			lcens <- ifelse((censor=="left"|censor=="both")&
				y[,1]==0,1,0)
			rcens <-ifelse((censor=="right"|censor=="both")&
				y[,2]==0,1,0)
			if(censor=="both"){
				lcens <- lcens/2
				rcens <- rcens/2}
			n <- nrow(y)
			nn <- y[,1]+y[,2]}
		else {
			if(any(delta<=0&y[,2]==1))
				stop("All deltas for uncensored data must be positive")
			else {
				delta <- ifelse(delta<=0,0.000001,delta)
				delta <- ifelse(y[,1]-delta/2<=0,delta-0.00001
				,delta)}
			y[,2] <- as.integer(y[,2])
			if(any(y[,2]!=-1&y[,2]!=0&y[,2]!=1))
				stop("Censor indicator must be -1, 0, or 1")
			if(censor!="left"&&censor!="right")
				stop("Censoring must be left or right")
			if(censor=="left"&!any(y[,2]==-1))
				stop("No left censored observations")
			if(censor=="right"&!any(y[,2]==0))
				stop("No right censored observations")
			cens <- as.integer(y[,2]==1)
			b <- as.integer((censor=="right"&y[,2]==0)|
				(censor=="left"&y[,2]==-1))
			r <- as.integer(censor=="left"&y[,2]==0)
			l <- as.integer(censor=="right"&y[,2]==-1)
			lc <- ifelse(censor=="left",1,0)
			rc <- ifelse(censor=="right",-1,1)}}
	if(dist=="double Poisson"||dist=="mult Poisson")
				my <- min(3*max(y),100)}
if((dist!="normal"&&dist!="logistic"&&dist!="Cauchy"&&
	dist!="Laplace"&&dist!="Student t"&&dist!="Poisson"&&
	dist!="negative binomial"&&dist!="Consul"&&dist!="double Poisson"&&
	dist!="mult Poisson"&&dist!="gamma count"&&dist!="binomial"&&
	dist!="beta binomial"&&dist!="double binomial"&&
	dist!="mult binomial")&&(any(y[,1]<=0)))
	stop("All response values must be > 0")
else if((dist=="Poisson"||dist=="negative binomial"||dist=="gamma count"||
		dist=="double Poisson"||dist=="mult Poisson"||dist=="Consul"||
		dist=="binomial"||dist=="beta binomial"||
		dist=="double binomial"||dist=="mult binomial")
		&&(any(y<0)))stop("All response values must be >= 0")
if(min(wt)<0)stop("All weights must be non-negative")
if(length(wt)==1)wt <- rep(wt,n)
if(length(delta)==1)delta <- rep(delta,n)
lin1 <- lin2 <- NULL
if(is.list(linear)){
	lin1 <- linear[[1]]
	lin2 <- linear[[2]]
	lmix <- is.language(lin2)}
else {
	lin1 <- linear
	lmix <- FALSE}
if(is.language(mu))lin1 <- mu
if(is.language(mix)){
	lin2 <- mix
	lmix <- is.language(lin2)}
nlp <- npl
if(is.language(lin1)){
	mt <- terms(lin1)
	if(is.numeric(mt[[2]])){
		dm1 <- matrix(1)
		colnames(dm1) <- "(Intercept)"
		npt1 <- 1
		if(!is.function(mu)){
			mu1 <- function(p) p[1]*rep(1,n)
			nlp <- 1}
		else mu1 <- function (p) mu(p, p[1]*rep(1,n))}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm1 <- model.matrix(mt, mf)
		npt1 <- ncol(dm1)
		if(!is.function(mu)){
			mu1 <- function(p) dm1 %*% p[1:npt1]
			nlp <- npt1}
		else mu1 <- function (p) mu(p, dm1 %*% p[1:npt1])}
	if(npl<npt1)stop("Not enough initial estimates for mu")}
else if(!is.function(mu)){
	mu1 <- function(p) p[1]*rep(1,n)
	nlp <- 1}
else mu1 <- mu
if(nlp!=npl)
	stop("Number of initial estimates for mu does not correspond to model")
npl1 <- npl+1
nlp <- npm
if(lmix){
	mt <- terms(lin2)
	if(is.numeric(mt[[2]])){
		dm2 <- matrix(1)
		colnames(dm2) <- "(Intercept)"
		npt2 <- 1
		if(!is.function(mix)) mixt <- function(p) {
			mf <- p[npl1]*rep(1,n)
			exp(mf)/(1+exp(mf))}
		else mixt <- function(p) {
			mf <- mix(p[npl1:np], p[npl1]*rep(1,n))
			exp(mf)/(1+exp(mf))}}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm2 <- model.matrix(mt, mf)
		npt2 <- ncol(dm2)
		if(!is.function(mix)){
			mixt <- function(p) {
				mf <- dm2 %*% p[npl1:(npl1+npt2-1)]
				exp(mf)/(1+exp(mf))}
			nlp <- npt2}
		else mixt <- function (p) {
			mf <- mix(p[npl1:np], dm2 %*% p[npl1:(nlp1+npt2-1)])
			exp(mf)/(1+exp(mf))}}
	if(npm<npt2)stop("Not enough initial estimates for mix")}
else if(!is.function(mix)){
	mixt <- function(p) exp(p[npl1])/(1+exp(p[npl1]))*rep(1,n)
	nlp <- 1}
else mixt <- function(p) {
	mf <- mix(p[npl1:np])
	exp(mf)/(1+exp(mf))}
if(nlp!=npm)
	stop("Number of initial estimates for mix does not correspond to model")
if(!is.numeric(mu1(pmu)))
	stop("The location function must return numerical values")
if(dist=="Levy"&&any(y[,1]<=mu1(p)))
	stop("location parameter must be strictly less than corresponding observation")
if(sht&&!is.numeric(mixt(p)))
	stop("The mix function must return numerical values")
ret <- switch(dist,
	binomial={
		fcn <- function(p) {
			m <- mu1(p)
			s <- mixt(p)
			-wt*log((1-s)*(lcens+rcens)+s*m^y[,1]*(1-m)^y[,2])}
		const <- -wt*(lchoose(nn,y[,1]))},
	"beta binomial"={
		fcn <- function(p) {
			m <- mu1(p)
			s <- mixt(p)
			v <- exp(p[np])
			t <- v*m
			u <- v*(1-m)
			-wt*log((1-s)*(lcens+rcens)+s*beta(y[,1]+t,y[,2]+u)/
				beta(t,u))}
		const <- -wt*(lchoose(nn,y[,1]))},
	"double binomial"={
		fcn <- function(p) {
			-wt*log((1-s)*(lcens+rcens)+s*exp(.C("ddb",
				as.integer(y[,1]),as.integer(nn),
				as.double(mu1(p)),as.double(exp(p[np])),
				as.integer(n),as.double(wt),
				res=double(n))$res))}
		const <- 0},
	"mult binomial"={
		fcn <- function(p) {
			-wt*log((1-s)*(lcens+rcens)+s*exp(.C("dmb",
				as.integer(y[,1]),as.integer(nn),
				as.double(mu1(p)),as.double(exp(p[np])),
				as.integer(n),as.double(wt),
				res=double(n))$res))}
		const <- 0},
	Poisson={
		fcn <- function(p) {
			m <- mu1(p)
			s <- mixt(p)
			-wt*log((1-s)*cens+s*exp(-m)*m^y)}
		const <- wt*lgamma(y+1)},
	"negative binomial"={
		fcn <- function(p) {
			m <- mu1(p)
			s <- mixt(p)
			t <- exp(p[np])
			-wt*log((1-s)*cens+s*gamma(y+t)/gamma(t)
				*t^t*m^y/(t+m)^(y+t))}
		const <- wt*lgamma(y+1)},
	"double Poisson"={
		fcn <- function(p) {
			-wt*log((1-s)*cens+s*exp(.C("ddp",as.integer(y),
				as.integer(my),as.double(mu1(p)),
				as.double(exp(p[np])),as.integer(length(y)),
				as.double(wt),res=double(length(y)))$res))}
		const <- 0},
	"mult Poisson"={
		fcn <- function(p) {
			-wt*log((1-s)*cens+s*exp(.C("dmp",as.integer(y),
				as.integer(my),as.double(mu1(p)),
				as.double(exp(p[np])),as.integer(length(y)),
				as.double(wt),res=double(length(y)))$res))}
		const <- 0},
	"gamma count"={
		fcn <- function(p) {
			m <- mu1(p)
			s <- mixt(p)
			t <- exp(p[np])
			-wt*log((1-s)*cens+s*ifelse(y==0,1-pgamma(m*t,
				(y+1)*t,1),pgamma(m*t,y*t+(y==0),1)-
				pgamma(m*t,(y+1)*t,1)))}
		const <- 0},
	Consul={
		fcn <- function(p) {
			m <- mu1(p)
			s <- mixt(p)
			u <- exp(p[np])
			-wt*log((1-s)*cens+s*m*exp(-(m+y*(u-1))/u-y*p[np])*
				(m+y*(u-1))^(y-1))}
		const <- wt*lgamma(y+1)},
	normal={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np]/2)
				pn <- pnorm(y[,1]-delta/2,m,t)
				-wt*log(s*cens*(pnorm(y[,1]+delta/2,m,t)-pn)
					+(1-cens)*((1+s*(rc*pn-lc))*b
					+s*(r+pn*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				pn <- pnorm(y[,1]-delta/2,m,exp(p[np]/2))
				-wt*log(s*cens*exp((-(p[np]+(y[,1]-m)^2
					*exp(-p[np]))/2))
					+(1-cens)*((1+s*(rc*pn-lc))*b
					+s*(r+pn*(l-r))))}
			const <- wt*cens*(log(2*pi)/2-log(delta))}},
        "inverse Gauss"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pit <- pinvgauss(y[,1]-delta/2,m,t)
				-wt*log(s*cens*(pinvgauss(y[,1]+delta/2,m,t)-pit)
					+(1-cens)*((1+s*(rc*pit-lc))*b
					+s*(r+pit*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pit <- pinvgauss(y[,1]-delta/2,m,t)
				-wt*log(s*cens*exp(-(p[np]+(y[,1]-m)^2/
					(y[,1]*t*m^2))/2)
					+(1-cens)*((1+s*(rc*pit-lc))*b
					+s*(r+pit*(l-r))))}
			const <- wt*cens*(log(2*pi*y[,1]^3)/2-log(delta))}},
	logistic={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])*sqrt(3)/pi
				pl <- plogis(y[,1]-delta/2,m,t)
				-wt*log(s*cens*(plogis(y[,1]+delta/2,m,t)-pl)
					+(1-cens)*((1+s*(rc*pl-lc))*b
					+s*(r+pl*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])*sqrt(3)/pi
				y1 <- (y[,1]-m)/t
				pl <- plogis(y[,1]-delta/2,m,t)
				-wt*log(s*cens*exp(-y1-log(t)
					-2*log(1+exp(-y1)))
					+(1-cens)*((1+s*(rc*pl-lc))*b
					+s*(r+pl*(l-r))))}
			const <- -wt*cens*log(delta)}},
        "Student t"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				ps <- pt(y[,1]-delta/2-m,t)
				-wt*log(s*cens*(pt(y[,1]+delta/2-m,t)-ps)
					+(1-cens)*((1+s*(rc*ps-lc))*b
					+s*(r+ps*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				ps <- pt(y[,1]-delta/2-m,t)
				-wt*log(s*cens*gamma((t+1)/2)/gamma(t/2)*
					exp(-p[np]/2-((t+1)/2)*
					log(1+(y[,1]-m)^2/t))
					+(1-cens)*((1+s*(rc*ps-lc))*b
					+s*(r+ps*(l-r))))}
			const <- wt*cens*(log(pi)/2-log(delta))}},
	Cauchy={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np]/2)
				pc <- pcauchy(y[,1]-delta/2,m,t)
				-wt*log(s*cens*(pcauchy(y[,1]+delta/2,m,t)-pc)
					+(1-cens)*((1+s*(rc*pc-lc))*b
					+s*(r+pc*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np]/2)
				pc <- pcauchy(y[,1]-delta/2,m,t)
				-wt*log(s*cens/(t*(1+(y[,1]-m)^2/t^2))
					+(1-cens)*((1+s*(rc*pc-lc))*b
					+s*(r+pc*(l-r))))}
			const <- -wt*cens*log(delta/pi)}},
        Laplace={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pl <- plaplace((y[,1]-delta/2-m)/t)
				-wt*log(s*cens*(plaplace((y[,1]+delta/2-m)/t)
					-pl)+(1-cens)*((1+s*(rc*pl-lc))*b
					+s*(r+pl*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pl <- plaplace((y[,1]-delta/2-m)/t)
				-wt*log(s*cens*exp(-abs(y[,1]-m)/t-p[np])+
					(1-cens)*((1+s*(rc*pl-lc))*b
					+s*(r+pl*(l-r))))}
			const <- -wt*cens*log(delta/2)}},
        Levy={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pl <- plevy(y[,1]-delta/2,m,t)
				-wt*log(s*cens*(plevy(y[,1]+delta/2,m,t)
					-pl)+(1-cens)*((1+s*(rc*pl-lc))*b
					+s*(r+pl*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pl <- plevy(y[,1]-delta/2,m,t)
				-wt*log(s*cens*sqrt(t/(2*pi))*log(y[,1]-m)^-1.5
					*exp(-t/(2*(y[,1]-m)))+(1-cens)*
					((1+s*(rc*pl-lc))*b+s*(r+pl*(l-r))))}
			const <- -wt*cens*log(delta/2)}},
        Pareto={
		if(exact){
			fcn <- function(p) {
				s <- mixt(p)
				u <- exp(p[np])
				t <- 1/(mu1(p)*u)
				pp <- (1+(y[,1]-delta/2)*t)^-u
				-wt*log(s*cens*(pp-(1+(y[,1]+delta/2)*t)^-u)
					+(1-cens)*((1+s*(rc*pp-lc))*b
					+s*(r+pp*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				s <- mixt(p)
				u <- exp(p[np])
				t <- 1/(mu1(p)*u)
				pp <- 1-(1+(y[,1]-delta/2)*t)^-u
				-wt*log(s*cens*u*t*(1+y[,1]*t)^(-(u+1))+
					(1-cens)*
					((1+s*(rc*pp-lc))*b+s*(r+pp*(l-r))))}
			const <- -wt*cens*log(delta)}},
	exponential={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				u <- exp(-(y[,1]-delta/2)/m)
				-wt*log(s*cens*(-exp(-(y[,1]+delta/2)/m)+u)
					+(1-cens)*((1+s*(rc*(1-u)-lc))*b
					+s*(r+(1-u)*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				u <- exp(-(y[,1]-delta/2)/m)
				-wt*log(s*cens*exp(-y[,1]/m)/m
					+(1-cens)*((1+s*(rc*(1-u)-lc))*b
					+s*(r+(1-u)*(l-r))))}
			const <- -wt*cens*log(delta)}},
        gamma={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				u <- m/t
				pg <- pgamma(y[,1]-delta/2,t,u)
				-wt*log(s*cens*(pgamma(y[,1]+delta/2,t,u)-pg)
					+(1-cens)*((1+s*(rc*pg-lc))*b
					+s*(r+pg*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				u <- m/t
				pg <- pgamma(y[,1]-delta/2,t,u)
				-wt*log(s*cens*y[,1]^(t-1)*exp(-y[,1]/u)/
					(u^t*gamma(t))
					+(1-cens)*((1+s*(rc*pg-lc))*b
					+s*(r+pg*(l-r))))}
			const <- -wt*cens*log(delta)}},
        Weibull={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pw <- pweibull(y[,1]-delta/2,t,m)
				-wt*log(s*cens*(pweibull(y[,1]+delta/2,t,m)-pw)
					+(1-cens)*((1+s*(rc*pw-lc))*b
					+s*(r+pw*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				v <- y[,1]/m
				u <- exp(-v^t)
				-wt*log(s*cens*t*v^(t-1)*u/m+
					(1-cens)*((1+s*(rc*(1-u)-lc))*b
					+s*(r+(1-u)*(l-r))))}
			const <- -wt*cens*log(delta)}},
        "extreme value"={
		if(exact){
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				pw <- pweibull(exp(y[,1])-delta/2,t,m)
				-wt*log(s*cens*(pweibull(exp(y[,1])+delta/2,
					t,m)-pw)+(1-cens)*((1+s*(rc*pw-lc))*b
					+s*(r+pw*(l-r))))}
			const <- 0}
		else {
			fcn <- function(p) {
				m <- mu1(p)
				s <- mixt(p)
				t <- exp(p[np])
				v <- exp(y[,1])/m
				u <- exp(-v^t)
				-wt*log(s*cens*t*v^(t-1)*u/m+
					(1-cens)*((1+s*(rc*(1-u)-lc))*b
					+s*(r+(1-u)*(l-r))))}
			const <- -wt*cens*log(delta)}},
	own={const <- 0})
fn <- function(p) sum(fcn(p))
if(fscale==1)fscale <- fn(p)
if(is.na(fscale))
	stop("Non-numerical function value: probably invalid initial values")
z0 <- nlm(fn, p=p, hessian=T, print.level=print.level, typsiz=typsiz,
	ndigit=ndigit, gradtol=gradtol, stepmax=stepmax, steptol=steptol,
	iterlim=iterlim, fscale=fscale)
z0$minimum <- z0$minimum+sum(const)
if(!is.language(lin1))cname <- paste("p",1:npl,sep="")
else {
     cname <- colnames(dm1)
     if(is.function(mu)&&length(cname)<npl)
	cname <- c(cname,paste("p",(length(cname)+1):npl,sep=""))}
if(!is.language(lin2))mname <- paste("p",1:npm,sep="")
else {
     mname <- colnames(dm2)
     if(is.function(mix)&&length(mname)<npm)
	mname <- c(mname,paste("p",(length(mname)+1):npm,sep=""))}
fitted.values <- if(dist=="binomial"||dist=="beta binomial"||
		dist=="double binomial"||dist=="mult binomial")
		as.vector((y[,1]+y[,2])*mu1(z0$estimate))
	else as.vector(mu1(z0$estimate))
residuals <- if(dist!="Poisson"&&dist!="negative binomial"&&dist!="Consul"
	&&dist!="double Poisson"&&dist!="mult Poisson"&&dist!="gamma count")
		y[,1]-fitted.values
	else y-fitted.values
if(np==1){
	cov <- 1/z0$hessian
	se <- sqrt(cov)}
else {
	a <- qr(z0$hessian)
	if(a$rank==np)cov <- solve(z0$hessian)
	else cov <- matrix(NA,ncol=np,nrow=np)
	se <- sqrt(diag(cov))}
like.comp <- as.vector(fcn(z0$estimate)+const)
if(is.function(mu))mu1 <- mu
if(is.function(mix))mixt <- mix
z1 <- list(
	call=call,
	delta=delta,
	dist=dist,
	likefn=fcn,
	mu=mu1,
	mix=mixt,
	linear=list(lin1,lin2),
	prior.weights=wt,
	censor=censor,
	maxlike=z0$minimum,
	fitted.values=fitted.values,
	residuals=residuals,
	like.comp=like.comp,
	aic=z0$minimum+np,
	df=sum(wt)-np,
	coefficients=z0$estimate,
	cname=cname,
	sname="p1",
	mname=mname,
	npl=npl,
	npm=npm,
	nps=as.numeric(sht),
	npf=0,
	se=se,
	cov=cov,
	corr=cov/(se%o%se),
	gradient=z0$gradient,
	iterations=z0$iterations,
	code=z0$code)
class(z1) <- "gnlr"
return(z1)}
