# LindleyGompertz.R
#' Lindley-Gompertz Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley-Gompertz distribution.
#'
#' The Lindley-Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley-Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda, \theta) =
#' \left( 1 - \exp\left\{ \frac{\lambda}{\alpha}
#' \left( 1 - \exp(\alpha x) \right) \right\} \right)^{\theta}
#' \left[ 1 - \frac{\theta}{1 + \theta}
#' \log\left\{ 1 - \exp\left( \frac{\lambda}{\alpha}
#' \left( 1 - \exp(\alpha x) \right) \right) \right\} \right],
#' \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' @usage
#' dlindley.gpz(x, alpha, lambda, theta, log = FALSE)
#' plindley.gpz(q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qlindley.gpz(p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rlindley.gpz(n, alpha, lambda, theta)
#' hlindley.gpz(x, alpha, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlindley.gpz}: numeric vector of (log-)densities
#'   \item \code{plindley.gpz}: numeric vector of probabilities
#'   \item \code{qlindley.gpz}: numeric vector of quantiles
#'   \item \code{rlindley.gpz}: numeric vector of random variates
#'   \item \code{hlindley.gpz}: numeric vector of hazard values
#' }
#'
#' @references 
#' Joshi, R. K., & Kumar, V. (2020). 
#' Lindley Gompertz distribution with properties and application. 
#' \emph{International Journal of Statistics and Applied Mathematics}, 
#' \bold{5(6)}, 28--37. \doi{10.22271/maths.2020.v5.i6a.610}  
#'
#' @examples
#' x <- seq(1, 10, 0.5)
#' dlindley.gpz(x, 0.1, 0.5, 1.5)
#' plindley.gpz(x, 0.1, 0.5, 1.5)
#' qlindley.gpz(0.5, 0.1, 0.5, 1.5)
#' rlindley.gpz(10, 0.1, 0.5, 1.5)
#' hlindley.gpz(x, 0.1, 0.5, 1.5)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(alpha=0.1765, lambda=0.2051, theta=11.4574)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plindley.gpz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlindley.gpz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlindley.gpz, pfun=plindley.gpz, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyGompertz
#' @aliases dlindley.gpz plindley.gpz qlindley.gpz rlindley.gpz hlindley.gpz
#' @export
NULL

#' @rdname LindleyGompertz
#' @usage NULL
#' @export   
dlindley.gpz <- function (x, alpha, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, lambda, theta) <= 0)) {
        stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- (lambda/alpha) * (1-exp(alpha*x[valid]))	
	z <- 1 -  exp(u) 
	w <- theta/(1+theta)
    pdf[valid] <- (theta*w) * lambda * exp(alpha*x[valid])*(z^(theta-1))*exp(u) * (1 - w*log(z))   
    }
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname LindleyGompertz
#' @usage NULL
#' @export 
plindley.gpz <- function (q, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda, theta) <= 0)) {
		stop("Parameters 'alpha', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- (lambda/alpha) * (1-exp(alpha*q[valid]))	
	z <- 1 -  exp(u) 
	w <- theta/(1+theta)	
    cdf[valid] <- (z ^ theta) * (1 - w * log(z))  
    }
	if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname LindleyGompertz
#' @usage NULL
#' @export 
qlindley.gpz <- function (p, alpha, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Parameters 'alpha', 'lambda', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	   theta *log(1-exp((lambda/alpha) *(1-exp(alpha*x)))) +
       log(1-(theta/(theta+1))*log(1-exp((lambda/alpha) *(1-exp(alpha*x)))))-log(wi)   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname LindleyGompertz
#' @usage NULL
#' @export 
rlindley.gpz <- function(n, alpha, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda, theta), is.numeric)))
		stop("'alpha', 'lambda', and 'theta' must be numeric.")
	if (any(c(alpha, lambda, theta) <= 0))
		stop("Invalid parameters: alpha, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qlindley.gpz(u, alpha, lambda, theta)
    return(q)
}

#' @rdname LindleyGompertz
#' @usage NULL
#' @export 
hlindley.gpz <- function (x, alpha, lambda, theta)
{
	if (!all(sapply(list(x, alpha, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
		
	nume <- dlindley.gpz(x, alpha, lambda, theta) 
	surv <- 1.0 - plindley.gpz(x, alpha, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
	
    return(hrf)   
} 

