# InvPham.R
#' Inverse Pham Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverse Pham distribution.
#'
#' The Inverse Pham distribution is parameterized by the parameters
#' \eqn{\beta > 0}, and \eqn{\delta > 0}.
#'
#' @details
#' The Inverse Pham distribution has CDF:
#'
#' \deqn{
#' F(x; \beta, \delta) =
#' \exp \left( {1 - {\delta ^{{x^{ - \beta }}}}} \right)
#' \quad ;\;x > 0.
#' }
#'
#' where\eqn{\beta} and \eqn{\delta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dinv.pham()} — Density function
#'   \item \code{pinv.pham()} — Distribution function
#'   \item \code{qinv.pham()} — Quantile function
#'   \item \code{rinv.pham()} — Random generation
#'   \item \code{hinv.pham()} — Hazard function
#' }
#'
#' @usage
#' dinv.pham(x, beta, delta, log = FALSE)
#' pinv.pham(q, beta, delta, lower.tail = TRUE, log.p = FALSE)
#' qinv.pham(p, beta, delta, lower.tail = TRUE, log.p = FALSE)
#' rinv.pham(n, beta, delta)
#' hinv.pham(x, beta, delta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param beta positive numeric parameter
#' @param delta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.pham}: numeric vector of (log-)densities
#'   \item \code{pinv.pham}: numeric vector of probabilities
#'   \item \code{qinv.pham}: numeric vector of quantiles
#'   \item \code{rinv.pham}: numeric vector of random variates
#'   \item \code{hinv.pham}: numeric vector of hazard values
#' }
#'
#' @references  
#' Elbatal, M., Araibi, M.I.A., Ocloo, S.K., Almetwally, E.M., Sapkota, L.P.,  
#' & Gemeay, A.M. (2025). Classical and Bayesian Methodology for a New Inverse
#' Statistical Model. \emph{Engineering Reports}, \bold{7(8)}, 1--33. 
#' \doi{10.1002/eng2.70323} 
#'   
#' Srivastava, A.K., & Kumar, V. (2011).  
#' Analysis of Pham (Loglog) Reliability Model Using Bayesian Approach. 
#' \emph{Computer Science Journal}, \bold{1(2)}, 79--100. 
#' 
#' Pham, H. (2002).  A Vtub-Shaped Hazard Rate Function With Applications
#' to System Safety. \emph{International Journal of Reliability and 
#' Applications}, \bold{3(1)}, 1--16.
#'
#' @examples
#' x <- seq(1, 10, 0.5)
#' dinv.pham(x, 0.5, 1.5)
#' pinv.pham(x, 0.5, 1.5)
#' qinv.pham(0.5, 0.5, 1.5)
#' rinv.pham(10, 0.5, 1.5)
#' hinv.pham(x, 0.5, 1.5)
#' 
#' # Data
#' x <- relief
#' # ML estimates
#' params = list(beta=2.8287, delta=9.6044)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.pham, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.pham, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dinv.pham, pfun=pinv.pham, plot=FALSE)
#' print.gofic(out)
#' 
#' @name InvPham
#' @aliases dinv.pham pinv.pham qinv.pham rinv.pham hinv.pham
#' @export
NULL

#' @rdname InvPham
#' @usage NULL
#' @export   
dinv.pham <- function (x, beta, delta, log = FALSE)
{
	if (!all(sapply(list(x, beta, delta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(beta, delta) <= 0)) {
		stop("Parameters 'beta' and 'delta' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- delta ^ (x[valid] ^ -beta)	
    pdf[valid] <- beta * log(delta) * (x[valid] ^ -(beta + 1)) * u * exp(1 - u)
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname InvPham
#' @usage NULL
#' @export 
pinv.pham <- function (q, beta, delta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, beta, delta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(beta, delta) <= 0)) {
		stop("Parameters 'beta' and 'delta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- delta ^ (q[valid] ^ -beta)
    cdf[valid] <- exp(1 - u)
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)   
}

#' @rdname InvPham
#' @usage NULL
#' @export 
qinv.pham <- function (p, beta, delta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, beta, delta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, delta) <= 0))
		stop("Invalid parameters: beta, and delta must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <-  (log(delta)/(log(1-log(p[valid])))) ^ (1/beta)      
    }
    return(qtl)  
}

#' @rdname InvPham
#' @usage NULL
#' @export 
rinv.pham <- function(n, beta, delta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(beta, delta), is.numeric)))
		stop("'beta', and 'delta' must be numeric.")
	if (any(c(beta, delta) <= 0))
		stop("Invalid parameters: beta, and delta must be positive.")

    u <- runif(n)
    q <- qinv.pham(u, beta, delta)
    return(q)
}

#' @rdname InvPham
#' @usage NULL
#' @export 
hinv.pham <- function (x, beta, delta)
{
    if (!all(sapply(list(x, beta, delta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, delta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dinv.pham(x, beta, delta, log=FALSE) 
    surv <- 1.0 - pinv.pham(x, beta, delta, lower.tail = TRUE, log.p = FALSE)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 




