# HLNHE.R
#' Half-Logistic NHE(Nadarajah-Haghighi Exponential) Distribution
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Logistic NHE distribution.
#'
#' The Half-Logistic NHE distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Half-Logistic NHE distribution has CDF:
#'
#' \deqn{
#' F(x;\,\alpha,\beta,\lambda) = 
#' \frac{{1 - \exp \left[ {\lambda \left\{ {1 - {{(1 + \alpha x)}^\beta }} 
#' \right\}} \right]}}{{1 + \exp \left[ {\lambda \left\{ {1 - {{(1 + \alpha x)}
#' ^\beta }} \right\}} \right]}} \, ;\quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dHL.nhe()} — Density function
#'   \item \code{pHL.nhe()} — Distribution function
#'   \item \code{qHL.nhe()} — Quantile function
#'   \item \code{rHL.nhe()} — Random generation
#'   \item \code{hHL.nhe()} — Hazard function
#' }
#'
#' @usage
#' dHL.nhe(x, alpha, beta, lambda, log = FALSE)
#' pHL.nhe(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qHL.nhe(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rHL.nhe(n, alpha, beta, lambda)
#' hHL.nhe(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dHL.nhe}: numeric vector of (log-)densities
#'   \item \code{pHL.nhe}: numeric vector of probabilities
#'   \item \code{qHL.nhe}: numeric vector of quantiles
#'   \item \code{rHL.nhe}: numeric vector of random variates
#'   \item \code{hHL.nhe}: numeric vector of hazard values
#' }
#'
#' @references  
#' Almarashi, A. M., Elgarhy, M., Elsehetry, M. M., Kibria, B. G., & Algarni, A. (2019). 
#' A new extension of exponential distribution with statistical properties
#' and applications. \emph{Journal of Nonlinear Sciences and Applications}, \bold{12}, 135--145.
#'
#' Chaudhary, A.K., & Kumar, V.(2020). 
#' Half Logistic Modified Exponential Distribution:Properties and Applications. 
#' \emph{EPRA International Journal of Multidisciplinary Research (IJMR)},
#' \bold{6(12)},276--286. \doi{10.36713/epra3291} 
#'  
#' Joshi, R. K., & Kumar, V. (2020). 
#' Half Logistic NHE: Properties and Application. \emph{International Journal
#' for Research in Applied Science & Engineering Technology (IJRASET)}, \bold{8(9)},
#' 742--753. \doi{10.22214/ijraset.2020.31557}
#' 
#' Nadarajah, S., & Haghighi, F. (2011). An extension of the exponential distribution. 
#' \emph{Statistics}, \bold{45(6)}, 543--558.
#' 
#' @examples
#' x <- seq(0.1, 1, 0.1)
#' dHL.nhe(x, 1.5, 0.8, 2)
#' pHL.nhe(x, 1.5, 0.8, 2)
#' qHL.nhe(0.5, 1.5, 0.8, 2)
#' rHL.nhe(10, 1.5, 0.8, 2)
#' hHL.nhe(x, 1.5, 0.8, 2)
#' 
#' #Data
#' x <- windshield
#' #ML Estimates    
#' params = list(alpha =0.1649, beta=3.7152, lambda=0.5881)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pHL.nhe, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qHL.nhe, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dHL.nhe, pfun = pHL.nhe, plot=FALSE)
#' print.gofic(out)
#' 
#' @name HLNHE
#' @aliases dHL.nhe pHL.nhe qHL.nhe rHL.nhe hHL.nhe
#' @export
NULL

#' @rdname HLNHE
#' @usage NULL
#' @export 
dHL.nhe <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- 1 + alpha * x[valid]	
	z <- exp(lambda * (1 - u ^ beta))			
    pdf[valid] <- 2 * alpha * beta * lambda * (u^(beta-1)) * z * ((1 + z) ^ -2)
    }
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname HLNHE
#' @usage NULL
#' @export 
pHL.nhe <- function (q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {	
	z <- exp(lambda * (1 - (1 + alpha * q[valid]) ^ beta))	
    cdf[valid] <- (1.0 - z)/(1.0 + z)
    }
	if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname HLNHE
#' @usage NULL
#' @export
qHL.nhe <- function (p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- (1/alpha) *((1- (1/lambda) *log((1.0 - p[valid])/(1.0 + p[valid]))) ^ (1.0/beta) -1)
    }
	return(qtl)     
}

#' @rdname HLNHE
#' @usage NULL
#' @export 
rHL.nhe <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qHL.nhe(u, alpha, beta, lambda)
    return(q)
}

#' @rdname HLNHE
#' @usage NULL
#' @export 
hHL.nhe <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
    nume <- dHL.nhe(x, alpha, beta, lambda, log = FALSE)
	surv <- 1.0 - pHL.nhe(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
