# HCChen.R
#' Half-Cauchy Chen Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Cauchy Chen distribution.
#'
#' The Half-Cauchy Chen distribution is parameterized by the parameters
#' \eqn{\beta > 0}, \eqn{\lambda > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Half-Cauchy Chen distribution has CDF:
#'
#' \deqn{
#' F(x; \beta, \lambda, \theta) =
#' \quad \frac{2}{\pi }\arctan \left\{ { - \frac{\lambda }{\theta }
#' (1 - {e^{{x^\beta }}})} \right\} \quad ;\;x > 0.
#' }
#'
#' where \eqn{\beta}, \eqn{\lambda}, and \eqn{\theta} are the parameters.
#'
#' Included functions are:
#' \itemize{
#'   \item \code{dhc.chen()} — Density function
#'   \item \code{phc.chen()} — Distribution function
#'   \item \code{qhc.chen()} — Quantile function
#'   \item \code{rhc.chen()} — Random generation
#'   \item \code{hhc.chen()} — Hazard function
#' }
#'
#' @usage
#' dhc.chen(x, beta, lambda, theta, log = FALSE)
#' phc.chen(q, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' qhc.chen(p, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
#' rhc.chen(n, beta, lambda, theta)
#' hhc.chen(x, beta, lambda, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dhc.chen}: numeric vector of (log-)densities
#'   \item \code{phc.chen}: numeric vector of probabilities
#'   \item \code{qhc.chen}: numeric vector of quantiles
#'   \item \code{rhc.chen}: numeric vector of random variates
#'   \item \code{hhc.chen}: numeric vector of hazard values
#' }
#'
#' @references 
#' Chaudhary, A.K., Yadav, R.S., & Kumar, V.(2023). 
#' Half-Cauchy Chen Distribution with Theories and Applications.\emph{ Journal of Institute  
#' of Science and Technology}, \bold{28(1)}, 45--55. \doi{10.3126/jist.v28i1.56494}
#'
#' Polson, N.G., & Scott, J.G. (2012). 
#' On the half-Cauchy prior for a global scale parameter. 
#' \emph{Bayesian Analysis}, \bold{7(4)}, 887--902.
#'
#' Telee, L.B.S., & Kumar, V.(2024). 
#' Arctan-Chen Distribution with Properties and Application. 
#' \emph{International Journal of Statistics and Reliability Engineering}, 
#' \bold{11(1)}, 93--100.
#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dhc.chen(x, 2.0, 0.5, 2.5)
#' phc.chen(x, 2.0, 0.5, 2.5)
#' qhc.chen(0.5, 2.0, 0.5, 2.5)
#' rhc.chen(10, 2.0, 0.5, 2.5)
#' hhc.chen(x, 2.0, 0.5, 2.5)
#' 
#' # Data
#' x <- conductors
#' # ML estimates
#' params = list(beta=0.9753, lambda=0.0398, theta=29.0272)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = phc.chen, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qhc.chen, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' res <- gofic(x, params = params,
#'              dfun = dhc.chen, pfun=phc.chen, plot=FALSE)
#' print.gofic(res)
#' @name HCChen
#' @aliases dhc.chen phc.chen qhc.chen rhc.chen hhc.chen
#' @export
NULL

#' @rdname HCChen
#' @usage NULL
#' @export   
dhc.chen <- function (x, beta, lambda, theta, log = FALSE)
{
    if (!all(sapply(list(x, beta, lambda, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(beta, lambda, theta) <= 0)) {
        stop("Parameters 'beta', 'lambda' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- x[valid] ^ beta		
	z <- lambda * (1.0 - exp(u))	
    pdf[valid] <- (2/pi)*(beta*lambda/theta)*exp(u) *(x^(beta-1)) * (1/(1+(-z/theta)^2))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname HCChen
#' @usage NULL
#' @export
phc.chen <- function (q, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, beta, lambda, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(beta, lambda, theta) <= 0)) {
		stop("Parameters 'beta', 'lambda' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- -(lambda/theta) * (1.0 - exp(q[valid] ^ beta))
    cdf[valid] <- (2/pi) * atan(u)  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname HCChen
#' @usage NULL
#' @export
qhc.chen <- function (p, beta, lambda, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, beta, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, lambda, theta) <= 0))
		stop("Invalid parameters: beta, 'lambda' and 'theta' must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    qtl[valid] <-  (log(1.0+ (theta/lambda)*tan(p[valid] * pi *0.5))) ^ (1/beta)	   
    }    
    return(qtl)  
 } 

#' @rdname HCChen
#' @usage NULL
#' @export
rhc.chen <- function(n, beta, lambda, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(beta, lambda, theta), is.numeric)))
		stop("'beta', 'lambda', and 'theta' must be numeric.")
	if (any(c(beta, lambda, theta) <= 0))
		stop("Invalid parameters: beta, 'lambda', and theta must be positive.")
  
    u <- runif(n)
    q <- qhc.chen(u, beta, lambda, theta)
    return(q)
}

#' @rdname HCChen
#' @usage NULL
#' @export
hhc.chen <- function (x, beta, lambda, theta)
{
	if (!all(sapply(list(x, beta, lambda, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(beta, lambda, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")    
    nume <- dhc.chen(x, beta, lambda, theta)
	surv <- 1.0 - phc.chen(x, beta, lambda, theta)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf) 
}



