# McCulloch quantile-based estimation for stable distributions
# =============================================================================
# TABLE GENERATION
# =============================================================================


#' Generate McCulloch lookup table from simulated stable samples
#'
#' Simulates alpha-stable samples across a grid of alpha and beta values, and computes
#' quantile-based ratios used for McCulloch estimation. The result is a lookup table
#' indexed by (alpha, beta) keys.
#'
#' @param alpha_grid Vector of alpha values to simulate.
#' @param beta_grid Vector of beta values to simulate.
#' @param size Number of samples per simulation.
#' @return Named list of quantile ratios indexed by "alpha_beta".
#' @importFrom stabledist rstable
#' @export
generate_mcculloch_table <- function(alpha_grid, beta_grid, size = 100000) {
  lookup <- list()
  for (alpha in alpha_grid) {
    for (beta in beta_grid) {
      X <- rstable(size, alpha, beta, gamma = 1, delta = 0)
      q <- compute_quantile_ratios(X)
      key <- paste(round(alpha, 2), round(beta, 2), sep = "_")
      lookup[[key]] <- list(v_alpha = q$v_alpha, v_beta = q$v_beta)
    }
  }
  return(lookup)
}

# =============================================================================
# INTERPOLATION FUNCTIONS
# =============================================================================


#' Build interpolation functions from McCulloch table
#'
#' Constructs nearest-neighbor interpolation functions for alpha and beta
#' based on quantile ratios (v_alpha, v_beta) from the lookup table.
#'
#' @param table Lookup table generated by generate_mcculloch_table.
#' @return List with functions interp_alpha and interp_beta.
#' @export
build_mcculloch_interpolators <- function(table) {
  points <- matrix(0, nrow = length(table), ncol = 2)
  alphas <- numeric(length(table))
  betas <- numeric(length(table))
  i <- 1
  for (key in names(table)) {
    parts <- strsplit(key, "_")[[1]]
    alphas[i] <- as.numeric(parts[1])
    betas[i] <- as.numeric(parts[2])
    points[i, ] <- c(table[[key]]$v_alpha, table[[key]]$v_beta)
    i <- i + 1
  }
  # Nearest neighbor interpolation (simplified)
  interp_alpha <- function(va, vb) {
    distances <- sqrt((points[, 1] - va)^2 + (points[, 2] - vb)^2)
    idx <- which.min(distances)
    return(alphas[idx])
  }
  interp_beta <- function(va, vb) {
    distances <- sqrt((points[, 1] - va)^2 + (points[, 2] - vb)^2)
    idx <- which.min(distances)
    return(betas[idx])
  }
  return(list(interp_alpha = interp_alpha, interp_beta = interp_beta))
}

# =============================================================================
# ESTIMATION FUNCTIONS
# =============================================================================

#' Estimate stable parameters using McCulloch lookup
#'
#' Estimates alpha and beta using quantile ratios and interpolation functions.
#' Gamma and delta are derived directly from quantiles.
#'
#' @param X Numeric vector of data.
#' @param interpolators Optional list with interp_alpha and interp_beta functions.
#' @param interp_alpha Optional function to interpolate alpha.
#' @param interp_beta Optional function to interpolate beta.
#' @return List with estimated alpha, beta, gamma, delta.
#' @export
mcculloch_lookup_estimate <- function(X, interpolators = NULL, interp_alpha = NULL, interp_beta = NULL) {
  q <- compute_quantile_ratios(X)

  # Handle both calling conventions
  if (!is.null(interpolators)) {
    # Called with interpolators object
    alpha <- interpolators$interp_alpha(q$v_alpha, q$v_beta)
    beta  <- interpolators$interp_beta(q$v_alpha, q$v_beta)
  } else if (!is.null(interp_alpha) && !is.null(interp_beta)) {
    # Called with separate functions
    alpha <- interp_alpha(q$v_alpha, q$v_beta)
    beta  <- interp_beta(q$v_alpha, q$v_beta)
  } else {
    stop("Must provide either 'interpolators' object or both 'interp_alpha' and 'interp_beta' functions")
  }

  gamma <- q$v_gamma
  delta <- -q$v_delta
  return(list(alpha = alpha, beta = beta, gamma = gamma, delta = delta))
}


#' Mock lookup for alpha and beta (fallback)
#'
#' Provides a fallback estimation of alpha and beta from quantile ratios
#' using simple linear approximations.
#'
#' @param v_alpha Quantile-based shape ratio.
#' @param v_beta Quantile-based skewness ratio.
#' @return List with estimated alpha and beta.
#' @export
mock_lookup_alpha_beta <- function(v_alpha, v_beta) {
  alpha <- pmax(pmin(3.5 - 1.5 * (v_alpha - 2), 2.0), 0.6)
  beta  <- pmax(pmin(v_beta, 1.0), -1.0)
  return(list(alpha = alpha, beta = beta))
}

#' Initialization using McCulloch quantile method
#'
#' Estimates all four stable parameters using quantile ratios and mock lookup.
#' Useful for initializing optimization algorithms.
#'
#' @param X Numeric vector of data.
#' @return List with estimated alpha, beta, gamma, delta.
#' @export
mcculloch_quantile_init <- function(X) {
  ratios <- compute_quantile_ratios(X)
  lookup_result <- mock_lookup_alpha_beta(ratios$v_alpha, ratios$v_beta)
  alpha <- lookup_result$alpha
  beta  <- lookup_result$beta
  gamma <- ratios$v_gamma
  delta <- -ratios$v_delta
  return(list(alpha = alpha, beta = beta, gamma = gamma, delta = delta))
}
