# Gradient computation functions for stable distribution parameter estimation

# =============================================================================
# PARAMETER VALIDATION
# =============================================================================


#' Validate and clip parameters for stable distribution
#'
#' Ensures parameters are within valid bounds for stable distribution computations.
#'
#' @param alpha Stability parameter (must be > 0 and < 2).
#' @param beta Skewness parameter (clipped to [-1, 1]).
#' @param delta Scale parameter (must be > 0).
#' @param r Optional numeric vector. Must be positive if provided.
#'
#' @return Clipped beta value.
#' @export
validate_params <- function(alpha, beta, delta, r = NULL) {
  if (delta <= 0 || alpha <= 0) {
    stop("delta > 0 and alpha > 0 are required")
  }

  if (alpha >= 2) {
    warning("alpha >= 2 may cause integration issues")
    alpha <- min(alpha, 1.99)
  }

  # FIX: Use any() to handle vector r properly
  if (!is.null(r) && any(r <= 0)) {
    stop("r > 0 is required")
  }

  beta <- pmax(pmin(beta, 1), -1)  # Clip to [-1, 1]
  return(beta)
}

# =============================================================================
# CORE INTEGRAND FUNCTIONS
# =============================================================================


#' Sine-weighted exponential with r^alpha term
#'
#' @param r Integration variable.
#' @param x Observation.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   ((0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric value of the integrand.
#' @export
sine_weighted_exp_ralpha <- function(r, x, alpha, beta, delta, omega) {
  beta <- validate_params(alpha, beta, delta, r)
  r_alpha <- r^alpha
  phase <- ((x - omega) / delta) * r - beta * tan(pi * alpha / 2) * r_alpha
  return(sin(phase) * r_alpha * exp(-pmin(r_alpha, 700)))
}

#' Sine-log-weighted exponential with r^(-alpha) term
#' @param r Integration variable.
#' @param x Observation.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric value of the integrand.
#' @export
sine_log_weighted_exp_ralpha <- function(r, x, alpha, beta, delta, omega) {
  beta <- validate_params(alpha, beta, delta, r)
  r_alpha <- r^alpha
  phase <- ((x - omega) / delta) * r - beta * tan(pi * alpha / 2) * r_alpha
  sin(phase) * log(r) * r^(-alpha) * exp(-pmin(r_alpha, 700))
}

#' Cosine-log-weighted exponential with r^(-alpha) term
#' @param r Integration variable.
#' @param x Observation.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric value of the integrand.
#' @export
cosine_log_weighted_exp_ralpha <- function(r, x, alpha, beta, delta, omega) {
  beta <- validate_params(alpha, beta, delta, r)

  # Add numerical stability checks
  if (any(r <= 0)) {
    return(rep(0, length(r)))
  }

  # Avoid very small r values that cause log(r) to be very negative
  r <- pmax(r, 1e-6)

  r_alpha <- r^alpha

  # More careful phase computation
  phase <- ((x - omega) / delta) * r - beta * tan(pi * alpha / 2) * r_alpha

  # Avoid overflow in exponential
  exp_term <- exp(-pmin(r_alpha, 700))

  # The problematic term: log(r) * r^(-alpha)
  # For small r, log(r) becomes very negative, r^(-alpha) becomes very large
  # We need to handle this more carefully

  log_r <- log(r)
  r_neg_alpha <- r^(-alpha)

  # Check for numerical issues
  log_r_term <- log_r * r_neg_alpha
  if (any(!is.finite(log_r_term))) {
    log_r_term[!is.finite(log_r_term)] <- 0
  }

  result <- cos(phase) * log_r_term * exp_term

  # Final check for finite values
  result[!is.finite(result)] <- 0

  return(result)
}


#' Cosine exponential function
#' @param r Integration variable.
#' @param x Observation.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric value of the integrand.
#' @export
cosine_exp_ralpha <- function(r, x, alpha, beta, delta, omega) {
  beta <- validate_params(alpha, beta, delta)
  r_alpha <- r^alpha
  phase <- ((x - omega) / delta) * r - beta * tan(pi * alpha / 2) * r_alpha
  cos(phase) * exp(-pmin(r_alpha, 700))
}

#' Sine exponential function
#' @param r Integration variable.
#' @param x Observation.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric value of the integrand.
#' @export
sine_exp_ralpha <- function(r, x, alpha, beta, delta, omega) {
  beta <- validate_params(alpha, beta, delta)
  r_alpha <- r^alpha
  phase <- ((x - omega) / delta) * r - beta * tan(pi * alpha / 2) * r_alpha
  sin(phase) * exp(-pmin(r_alpha, 700))
}

#' Sine-r-weighted exponential function
#' @param r Integration variable.
#' @param x Observation.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric value of the integrand.
#' @export
sine_r_weighted_exp_ralpha <- function(r, x, alpha, beta, delta, omega) {
  beta <- validate_params(alpha, beta, delta)
  r_alpha <- r^alpha
  phase <- ((x - omega) / delta) * r - beta * tan(pi * alpha / 2) * r_alpha
  sin(phase) * r * exp(-pmin(r_alpha, 700))
}

# =============================================================================
# ROBUST INTEGRATION FUNCTIONS
# =============================================================================


#' Safe integration wrapper with multiple fallback strategies
#'
#' Performs numerical integration with progressively more conservative
#' settings to improve robustness. Useful when standard integration
#' may fail due to oscillatory or heavy-tailed functions.
#'
#' @param integrand_func Function to integrate. Must accept a numeric vector
#'   as input and return numeric values.
#' @param lower Numeric scalar. Lower bound of the integration interval
#'   (default = 1e-8 to avoid singularities at 0).
#' @param upper Numeric scalar. Upper bound of the integration interval
#'   (default = 50).
#' @param max_attempts Integer. Maximum number of fallback attempts with
#'   modified settings if the initial integration fails.
#' @param ... Additional arguments passed to \code{integrand_func}.
#'
#' @return Numeric scalar. The estimated value of the integral, or 0 if all
#'   attempts fail.
#' @importFrom stats integrate
#' @importFrom utils head tail
#' @export
safe_integrate <- function(integrand_func, lower = 1e-8, upper = 50,
                           max_attempts = 3, ...) {

  # Try progressively more conservative approaches
  upper_limits <- c(upper, upper/2, upper/5, 10, 5)
  tolerances <- list(
    list(rel.tol = 1e-6, abs.tol = 1e-8),
    list(rel.tol = 1e-4, abs.tol = 1e-6),
    list(rel.tol = 1e-3, abs.tol = 1e-5)
  )

  for (ul in upper_limits) {
    for (tol in tolerances) {
      result <- tryCatch({
        # Test integrand at a few points first
        test_points <- c(lower, (lower + ul)/2, ul * 0.9)
        for (tp in test_points) {
          test_val <- integrand_func(tp)
          if (!is.finite(test_val) || abs(test_val) > 1e10) {
            stop("Problematic integrand")
          }
        }

        int_result <- integrate(integrand_func, lower, ul,
                                rel.tol = tol$rel.tol,
                                abs.tol = tol$abs.tol,
                                stop.on.error = FALSE, ...)

        if (int_result$message == "OK" && is.finite(int_result$value)) {
          return(int_result$value)
        } else {
          stop("Integration not OK")
        }

      }, error = function(e) NULL)

      if (!is.null(result)) return(result)
    }
  }

  # All attempts failed
  warning("All integration attempts failed - returning 0")
  return(0)
}


#' Robust integration helper function
#'
#' Applies `safe_integrate` to a given integrand over a vector of x values.
#'
#' @param f Function. The integrand function to be evaluated.
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#' @param upper Numeric scalar. Upper limit of integration (default = 50).
#' @param eps Numeric scalar. Tolerance for numerical integration (default = 1e-8).
#'
#' @return Numeric vector of integral values evaluated at each element of \code{x}.
#' @export
integrate_function <- function(f, x, alpha, beta, delta, omega,
                               upper = 50, eps = 1e-8) {

  integrate_single <- function(xi) {
    # Validate parameters first
    if (!is.finite(alpha) || !is.finite(beta) || !is.finite(delta) ||
        !is.finite(omega) || !is.finite(xi)) {
      warning("Non-finite parameters detected")
      return(0)
    }

    if (alpha <= 0 || alpha >= 2 || abs(beta) > 1 || delta <= 0) {
      warning("Parameters out of valid range")
      return(0)
    }

    # Use safe_integrate instead of basic integrate
    result <- safe_integrate(function(r) f(r, xi, alpha, beta, delta, omega),
                             lower = eps, upper = upper)
    return(result)
  }

  if (length(x) > 1) {
    sapply(x, integrate_single)
  } else {
    integrate_single(x)
  }
}

# =============================================================================
# INTEGRATION WRAPPER FUNCTIONS
# =============================================================================

#' Integration wrappers for specific integrands
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric vector of integral values.
#' @export
integrate_sine_weighted <- function(x, alpha, beta, delta, omega) {
  integrate_function(sine_weighted_exp_ralpha, x, alpha, beta, delta, omega)
}


#' Integrate sine-log-weighted exponential
#'
#' Computes the integral of the sine-log-weighted exponential function for stable distributions.
#'
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric vector of integral values.
#' @export
integrate_sine_log_weighted <- function(x, alpha, beta, delta, omega) {
  integrate_function(sine_log_weighted_exp_ralpha, x, alpha, beta, delta, omega)
}



#' Integrate cosine-log-weighted exponential
#'
#' Computes the integral of the cosine-log-weighted exponential function for stable distributions.
#'
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric vector of integral values.
#' @export
integrate_cosine_log_weighted <- function(x, alpha, beta, delta, omega) {
  integrate_single <- function(xi) {
    # More conservative integration bounds for this difficult integrand
    eps <- max(1e-4, 0.01^(1/alpha))  # Avoid very small r values
    upper <- min(20, 10^(2/alpha))    # More conservative upper limit

    # Try with different approaches
    approaches <- list(
      list(lower = eps, upper = upper/4),
      list(lower = eps, upper = upper/10),
      list(lower = max(eps, 0.1), upper = min(upper, 5))
    )

    for (approach in approaches) {
      result <- tryCatch({
        int_result <- integrate(
          function(r) cosine_log_weighted_exp_ralpha(r, xi, alpha, beta, delta, omega),
          lower = approach$lower,
          upper = approach$upper,
          rel.tol = 1e-4,
          abs.tol = 1e-6,
          subdivisions = 1000
        )

        if (int_result$message == "OK" && is.finite(int_result$value)) {
          return(int_result$value)
        } else {
          NULL
        }
      }, error = function(e) NULL)

      if (!is.null(result)) return(result)
    }

    # If all approaches fail, return 0 with warning
    warning(paste("Integration failed for x =", xi))
    return(0)
  }

  if (length(x) > 1) {
    sapply(x, integrate_single)
  } else {
    integrate_single(x)
  }
}



#' Integrate cosine exponential
#'
#' Computes the integral of the cosine exponential function for stable distributions.
#'
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric vector of integral values.
#' @export
integrate_cosine <- function(x, alpha, beta, delta, omega) {
  integrate_function(cosine_exp_ralpha, x, alpha, beta, delta, omega)
}



#' Integrate sin exponential
#'
#' Computes the integral of the sin exponential function for stable distributions.
#'
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric vector of integral values.
#' @export
integrate_sine <- function(x, alpha, beta, delta, omega) {
  integrate_function(sine_exp_ralpha, x, alpha, beta, delta, omega)
}



#' Integrate sine-r-weighted exponential
#'
#' Computes the integral of the sine-r-weighted exponential function for stable distributions.
#'
#' @param x Numeric vector. Observations at which the integral is computed.
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#'
#' @return Numeric vector of integral values.
#' @export
integrate_sine_r_weighted <- function(x, alpha, beta, delta, omega) {
  integrate_function(sine_r_weighted_exp_ralpha, x, alpha, beta, delta, omega)
}

# =============================================================================
# GRADIENT FUNCTIONS
# =============================================================================

#' Log-likelihood gradient with respect to alpha
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#' @param x Numeric vector. Observations at which the integral is computed.
#' @return Numeric scalar: estimated gradient (or \code{NA_real_} on invalid input).
#' @export
grad_loglik_alpha <- function(alpha, beta, delta, omega, x) {
  # Validate parameters first
  if (!is.finite(alpha) || !is.finite(beta) || !is.finite(delta) ||
      !is.finite(omega) || any(!is.finite(x))) {
    return(0)
  }

  if (alpha <= 0 || alpha >= 2 || abs(beta) > 1 || delta <= 0) {
    return(0)
  }

  term1 <- (pi * beta / (2 * cos(pi * alpha / 2)^2)) *
    integrate_function(sine_weighted_exp_ralpha, x, alpha, beta, delta, omega)

  return(term1 / (pi * delta))
}

#' Log-likelihood gradient with respect to beta
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#' @param x Numeric vector. Observations at which the integral is computed.
#' @return Numeric scalar: estimated gradient (or \code{NA_real_} on invalid input).
#' @export
grad_loglik_beta <- function(alpha, beta, delta, omega, x) {
  # Validate parameters first
  if (!is.finite(alpha) || !is.finite(beta) || !is.finite(delta) ||
      !is.finite(omega) || any(!is.finite(x))) {
    return(0)
  }

  if (alpha <= 0 || alpha >= 2 || abs(beta) > 1 || delta <= 0) {
    return(0)
  }

  result <- safe_integrate(function(r) sine_weighted_exp_ralpha(r, x, alpha, beta, delta, omega))
  return(result * tan(pi * alpha / 2) / (pi * delta))
}

#' Log-likelihood gradient with respect to delta (scale)
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#' @param x Numeric vector. Observations at which the integral is computed.
#' @return Numeric scalar: estimated gradient (or \code{NA_real_} on invalid input).
#' @export
grad_loglik_delta <- function(alpha, beta, delta, omega, x) {
  # Validate parameters first
  if (!is.finite(alpha) || !is.finite(beta) || !is.finite(delta) ||
      !is.finite(omega) || any(!is.finite(x))) {
    return(0)
  }

  if (alpha <= 0 || alpha >= 2 || abs(beta) > 1 || delta <= 0) {
    return(0)
  }

  term1 <- -1 / (pi * delta^2) *
    safe_integrate(function(r) cosine_exp_ralpha(r, x, alpha, beta, delta, omega))

  term2 <- ((x - omega) / (pi * delta^3)) *
    safe_integrate(function(r) sine_r_weighted_exp_ralpha(r, x, alpha, beta, delta, omega))

  return(term1 + term2)
}

#' Log-likelihood gradient with respect to omega (location)
#' @param alpha Numeric scalar. Stability parameter of the stable distribution
#'   (0 < \eqn{\alpha \le 2}), controlling tail thickness.
#' @param beta Numeric scalar. Skewness parameter of the stable distribution
#'   (\eqn{-1 \le \beta \le 1}).
#' @param delta Numeric scalar. Location parameter of the stable distribution.
#' @param omega Numeric scalar. Scale parameter of the stable distribution
#'   (omega > 0).
#' @param x Numeric vector. Observations at which the integral is computed.
#' @return Numeric scalar: estimated gradient (or \code{NA_real_} on invalid input).
#' @export
grad_loglik_omega <- function(alpha, beta, delta, omega, x) {
  # Validate parameters first
  if (!is.finite(alpha) || !is.finite(beta) || !is.finite(delta) ||
      !is.finite(omega) || any(!is.finite(x))) {
    return(0)
  }

  if (alpha <= 0 || alpha >= 2 || abs(beta) > 1 || delta <= 0) {
    return(0)
  }

  result <- 1 / (pi * delta^2) *
    safe_integrate(function(r) sine_r_weighted_exp_ralpha(r, x, alpha, beta, delta, omega))

  return(result)
}

# =============================================================================
# NORMALIZED GRADIENT FUNCTIONS
# =============================================================================

#' Normalized gradient for alpha parameter
#' Computes the normalized gradient of the log-likelihood with respect to the alpha parameter
#' over a set of observations. This is useful for optimization routines where scale-invariant
#' updates are preferred.
#'
#' @param alpha Stability parameter.
#' @param beta Skewness parameter.
#' @param delta Scale parameter.
#' @param omega Location parameter.
#' @param x Numeric vector of observations.
#' @return Scalar value representing the normalized gradient.
normalized_grad_alpha <- function(alpha, beta, delta, omega, x) {
  grad_vals <- sapply(x, function(xi) grad_loglik_alpha(alpha, beta, delta, omega, xi))
  pdf_vals <- pmax(r_stable_pdf(x, alpha, beta, delta, omega), 1e-300)
  sum(grad_vals) / sum(pdf_vals)
}


#' Normalized objective for beta parameter
#'
#' Computes the normalized gradient of the log-likelihood with respect to the beta parameter
#' using the latest values of alpha, delta, and omega stored in global vectors.
#'
#' @param beta Skewness parameter.
#' @param X1 Numeric vector of observations.
#' @param L_alpha Vector of alpha values (global memory).
#' @param L_delta Vector of delta values (global memory).
#' @param L_omega Vector of omega values (global memory).
#' @return Scalar value representing the normalized gradient.
normalized_objective_beta <- function(beta, X1, L_alpha, L_delta, L_omega) {
  alpha <- tail(L_alpha, 1)
  delta <- tail(L_delta, 1)
  omega <- tail(L_omega, 1)
  pdf_vals <- pmax(r_stable_pdf(X1, alpha, beta, delta, omega), 1e-300)
  grad_vals <- sapply(X1, function(xi) grad_loglik_beta(alpha, beta, delta, omega, xi))
  sum(grad_vals) / sum(pdf_vals)
}


#' Normalized objective for delta parameter
#'
#' Computes the normalized gradient of the log-likelihood with respect to the delta parameter
#' using the latest values of alpha, beta, and omega stored in global vectors.
#'
#' @param delta Location parameter.
#' @param X1 Numeric vector of observations.
#' @param L_alpha Vector of alpha values.
#' @param L_beta Vector of beta values.
#' @param L_omega Vector of omega values.
#' @return Scalar value representing the normalized gra
normalized_objective_delta <- function(delta, X1, L_alpha, L_beta, L_omega) {
  alpha <- tail(L_alpha, 1)
  beta <- tail(L_beta, 1)
  omega <- tail(L_omega, 1)
  pdf_vals <- pmax(r_stable_pdf(X1, alpha, beta, delta, omega), 1e-300)
  grad_vals <- sapply(X1, function(xi) grad_loglik_delta(alpha, beta, delta, omega, xi))
  sum(grad_vals) / sum(pdf_vals)
}



#' Normalized objective for omega parameter
#'
#' Computes the normalized gradient of the log-likelihood with respect to the omega parameter
#' using the latest values of alpha, beta, and delta stored in global vectors.
#'
#' @param omega Location parameter.
#' @param X1 Numeric vector of observations.
#' @param L_alpha Vector of alpha values.
#' @param L_beta Vector of beta values.
#' @param L_delta Vector of delta values.
#' @return Scalar value representing the normalized gradient.
normalized_objective_omega <- function(omega, X1, L_alpha, L_beta, L_delta) {
  alpha <- tail(L_alpha, 1)
  beta <- tail(L_beta, 1)
  delta <- tail(L_delta, 1)
  pdf_vals <- pmax(r_stable_pdf(X1, alpha, beta, delta, omega), 1e-300)
  grad_vals <- sapply(X1, function(xi) grad_loglik_omega(alpha, beta, delta, omega, xi))
  sum(grad_vals) / sum(pdf_vals)
}

# =============================================================================
# OPTIMIZATION HELPER FUNCTIONS
# =============================================================================

#' False position method update step
#'
#' Performs a single iteration of the false position (regula falsi) method
#' for root-finding. This method approximates the root of a function by
#' using a linear interpolation between two points where the function
#' changes sign.
#'
#' @param a Numeric scalar. Left bound of the interval.
#' @param b_n Numeric scalar. Right bound of the interval (current step).
#' @param f_a Numeric scalar. Value of the objective function evaluated at `a`.
#' @param f_b Numeric scalar. Value of the objective function evaluated at `b_n`.
#' @param objective_func Function. The objective function whose root is being sought.
#'
#' @return Numeric scalar. Updated estimate for the root after one iteration.
#' @export
false_position_update <- function(a, b_n, f_a, f_b, objective_func) {
  if (abs(f_a - f_b) < 1e-10) return(a)

  c <- (a * f_b - b_n * f_a) / (f_b - f_a)
  if (!is.finite(c)) return(a)

  f_c <- tryCatch(objective_func(c), error = function(e) return(NA))
  if (is.na(f_c)) return(a)

  if (f_c * f_a < 0) return(c) else return(c)
}
