#' Test for a single the median absolute deviation (MAD)
#' @description
#' Calculates a confidence interval and test the null hypothesis for the median absolute deviation (MAD) for a single sample.
#'
#' @param x a numeric vector of data values.
#' @param constant a scale factor. Default choice ensures population mad is equal to sd for normal distributions.
#' @param alternative a character string specifying the alternative hypothesis to be used. Needs to be one of "two.sided" (default), "greater" or "less".
#' @param conf.level confidence level for the confidence interval.  The default is 0.95 (for a 95% confidence interval).
#' @param true.mad a number which is the nominated value for the population mad under the null hypothesis.  Default is 1.
#' @param use.gld a logical indicating whether the Generalised Lambda Distribution (GLD) should be used to estimate the density of the data.  Default is FALSE in which case the function `density` is used.
#' @param gld.est a character string indicating which gld estimator to use if `use.gld = TRUE`.  See details below.
#' @param ... additional arguments, if desired, to be passed to function density for density estimation (see help file for `density()` for more details).
#'
#' @return Hypothesis test results comparing two mads from independent samples and associated confidence interval for the ratio of mads (a list with class "htest").  This list includes values
#'\describe{
#' \item{\code{statistic}}{the value of the Z-statistic.}
#' \item{\code{p.value}}{the p-value for the test.}
#' \item{\code{conf.int}}{a confidence interval for the MAD that is appropriate for the alternative hypothesis.}
#' \item{\code{estimate}}{the estimated MAD.}
#' \item{\code{null.value}}{the specified hypothesized value of the MAD under the null hypothesis.}
#' \item{\code{alternative}}{a character string describing the alternative hypothesis.}
#' \item{\code{data.name}}{a character string giving the names of the data.}
#' }
#' @details
#' This function calculates an approximate confidence interval and conducts a hypothesis test for a single mad. The approximate variance used in the construction of the confidence interval and the test statistic is calculated using the `varmad()` function (see Arachchige and Prendergast (2024) and the help file for `varmad()` for further details).
#' Note that the interval and test statistic are approximate and may be unreliable for small sample sizes.  Arachchige and Prendergast (2024) showed that reliable results
#' can be achieved for sample sizes of 50 or more when simulating data from various distributions.  Additionally, the confidence interval is a Wald-type interval (estimate +/- crit*SE)
#' and so it may be possible that the domain of the interval contains negative values.  Since negative values are not possible for the mad, thought could be given to truncation (setting the lower bound to zero), although
#' this may undermine interval coverage.
#'
#' @references
#' Arachchige, C.N.P.G., & Prendergast, L.A. (2026) Confidence intervals for median absolute deviations. Communications in Statistics-Simulation and Computation, 55(1), 13-22.
#' @export
#'
#' @examples
#' # Create some data
#' set.seed(1234)
#' x <- rlnorm(100)
#'
#' # Calculate the mad, 95% confidence interval and test the hypothesis
#' mad.est <- madci(x)
#' mad.est

madci <- function(x, constant = 1.4826, alternative = c("two.sided", "less", "greater"),
                  conf.level = 0.95, true.mad = 1,  use.gld = FALSE, gld.est = "TM", ...)
{
  if (!is.numeric(x))
    stop("Argument 'x' must be numeric.")

  dname <- deparse(substitute(x))

  if (anyNA(x)) {
    count.x.na <- sum(is.na(x))
    warning(paste0(count.x.na), " missing values removed in ",
            deparse(substitute(x)), ".\n")
    x <- stats::na.omit(x)
  }

  alternative <- match.arg(alternative)

  alpha <- 1 - conf.level
  crit <- stats::qnorm(1 - alpha/2)
  n <-length(x)

  est <- stats::mad(x, constant = constant)
  names(est) <- "Median Absolute Deviation"
  est.var <- varmad(x, constant = constant,  use.gld = use.gld, gld.est = gld.est, ...)
  sterr <- sqrt(est.var)

  test.stat <- (est - true.mad)/sterr
  names(test.stat) <- "Z"

  if (alternative == "less") {
    pval <- stats::pnorm(test.stat)
    ci <- c(0, est + stats::qnorm(conf.level) * sterr)
  }
  else if (alternative == "greater") {
    pval <- stats::pnorm(test.stat, lower.tail = FALSE)
    ci <- c(est - stats::qnorm(conf.level) * sterr, Inf)
  }
  else {
    pval <- 2 * (1 - stats::pnorm(abs(test.stat)))
    ci <- est + c(-1, 1) * crit * sterr
  }
  attr(ci, "conf.level") <- conf.level
  names(true.mad) <- names(est)
  res <- list(data.name = dname, statistic = test.stat, parameter = NULL, p.value = pval,
              alternative = alternative, estimate = est, null.value = true.mad, conf.int = ci)
  class(res) <- "htest"
  return(res)

}
