% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reliability.R
\name{reliability}
\alias{reliability}
\title{Estimate scale reliability for Likert and rating-scale data}
\usage{
reliability(
  data,
  include = "none",
  ci = FALSE,
  ci_level = 0.95,
  n_boot = 1000,
  na_method = c("pairwise", "listwise"),
  min_count = 2,
  digits = 3,
  verbose = TRUE
)
}
\arguments{
\item{data}{A data frame or matrix containing item responses.
Each column represents one item; rows represent respondents.}

\item{include}{Character vector specifying which additional estimates
to compute. Possible values are:
\itemize{
\item \code{"none"} (default): Pearson-based alpha and omega only.
\item \code{"lambda6"}: Include Guttman’s lambda-6 (requires package \pkg{psych}).
\item \code{"polychoric"}: Include ordinal (polychoric-based) alpha and omega.
}
Multiple options may be supplied.}

\item{ci}{Logical; if \code{TRUE}, confidence intervals are computed using
nonparametric bootstrap. Default is \code{FALSE}.}

\item{ci_level}{Confidence level for bootstrap intervals.
Default is \code{0.95}.}

\item{n_boot}{Number of bootstrap resamples used when \code{ci = TRUE}.
Default is \code{1000}.}

\item{na_method}{Method for handling missing values. Either
\code{"pairwise"} (default) or \code{"listwise"}.}

\item{min_count}{Minimum observed frequency per response category required
to attempt polychoric correlations. Ordinal reliability estimates are
skipped if this condition is violated. Default is \code{2}.}

\item{digits}{Number of decimal places used when printing estimates.
Default is \code{3}.}

\item{verbose}{Logical; if \code{TRUE}, warnings and progress indicators
are displayed. Default is \code{TRUE}.}
}
\value{
A tibble with one row per reliability coefficient and columns:
\itemize{
\item \code{coef_name}: Name of the reliability coefficient.
\item \code{estimate}: Point estimate.
\item \code{ci_lower}, \code{ci_upper}: Confidence interval bounds
(only present when \code{ci = TRUE}).
\item \code{notes}: Methodological notes describing how the estimate
was obtained.
}

The returned object has class \code{"likert_reliability"} and includes
additional attributes containing diagnostics and bootstrap information.
}
\description{
Computes internal consistency reliability estimates for a single-factor
scale, including Cronbach’s alpha, McDonald’s omega (total), and optional
ordinal (polychoric-based) variants. Confidence intervals may be obtained
via nonparametric bootstrap.
}
\details{
The function is designed for Likert-type and rating-scale data and
prioritises transparent diagnostics when ordinal reliability estimates
are not feasible due to sparse response categories.

Cronbach’s alpha and McDonald’s omega are computed from Pearson correlations.
When \code{include = "polychoric"}, ordinal reliability estimates are computed
using polychoric correlations and correspond to Zumbo’s alpha and ordinal omega.

Ordinal reliability estimates are skipped if response categories are sparse
or if polychoric estimation fails. Diagnostics explaining these decisions
are stored in the returned object and may be inspected using
\code{\link{ordinal_diagnostics}}.

This function assumes a single common factor and is not intended for
multidimensional or structural equation modelling contexts.
}
\examples{

## create dataset
my_cor <- LikertMakeR::makeCorrAlpha(
  items = 4,
  alpha = 0.80
)

my_data <- LikertMakeR::makeScales(
  n = 64,
  means = c(2.75, 3.00, 3.25, 3.50),
  sds = c(1.25, 1.50, 1.30, 1.25),
  lowerbound = rep(1, 4),
  upperbound = rep(5, 4),
  cormatrix = my_cor
)

## run function
reliability(my_data)

reliability(
  my_data,
  include = c("lambda6", "polychoric")
)

\donttest{
## slower (not run on CRAN checks)
reliability(
  my_data,
  include = "polychoric",
  ci = TRUE,
  n_boot = 200
)
}

}
\seealso{
\code{\link{ordinal_diagnostics}}
}
