# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_unemployment

library(testthat)

test_that("get_japan_unemployment() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_unemployment()

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

test_that("get_japan_unemployment() returns correct dimensions and years", {
  skip_on_cran()

  result <- get_japan_unemployment()

  # Number of rows: 2010–2022 inclusive
  expect_equal(nrow(result), 13)

  # Exact years
  expect_equal(sort(result$year), 2010:2022)

  # Should be sorted in descending order
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_unemployment() returns consistent values for Japan", {
  skip_on_cran()

  result <- get_japan_unemployment()

  # Country should always be Japan
  expect_true(all(result$country == "Japan"))

  # Indicator should always match unemployment
  expect_true(all(grepl("Unemployment", result$indicator, ignore.case = TRUE)))
  expect_true(all(grepl("labor force", result$indicator, ignore.case = TRUE)))

  # Values should be positive
  expect_true(all(result$value > 0, na.rm = TRUE))

  # Japan unemployment historically < 10%
  expect_true(all(result$value < 10, na.rm = TRUE))
})

test_that("get_japan_unemployment() has valid numeric values and no missing data", {
  skip_on_cran()

  result <- get_japan_unemployment()

  # No missing values
  expect_true(all(!is.na(result$indicator)))
  expect_true(all(!is.na(result$country)))
  expect_true(all(!is.na(result$year)))
  expect_true(all(!is.na(result$value)))

  # Year must be integer
  expect_true(is.integer(result$year))

  # Value must be numeric
  expect_true(is.numeric(result$value))
})

test_that("get_japan_unemployment() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_unemployment(), error = function(e) NULL)

  # Should return NULL (if API fails) or a tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
