% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shock_cal.R
\name{shock_calculate}
\alias{shock_calculate}
\title{Calculate Shocks from Initial and Target Values}
\usage{
shock_calculate(
  initial_config,
  target_config,
  calc_config,
  output_path,
  long_desc = "Calculated shock values",
  dim_order = NULL,
  lowercase = FALSE
)
}
\arguments{
\item{initial_config}{A list created by \code{\link{create_initial_config}}, defining:
\itemize{
\item Input path, file format, and variable header for the initial dataset
\item Column name of the initial value field ("Value_ini")
}}

\item{target_config}{A list created by \code{\link{create_target_config}}, defining:
\itemize{
\item Path, format, and variable header for the target dataset
\item Target value column ("Value_tar") or numeric target for uniform shock
}}

\item{calc_config}{A list created by \code{\link{create_calc_config}}, specifying:
\itemize{
\item \code{column_mapping}: mapping between initial and target columns
\item \code{timeline}: sequence of years or periods (e.g., "1-5")
\item \code{exclude_self_trade}: logical, whether to drop self-pairs
\item \code{exclusion_values}: list of region/sector values to omit
}}

\item{output_path}{Path to the output HAR file where calculated shocks will be written.}

\item{long_desc}{Optional text for header description. Default is "Calculated shock values".}

\item{dim_order}{Optional dimension ordering specification. Can be:
\itemize{
\item NULL (default): alphabetical A-Z ordering
\item a named list defining order for each dimension (e.g., REG, COMM)
\item a data frame or path to Excel/CSV file containing order definitions
}}

\item{lowercase}{Logical; if TRUE, converts dimension elements to lowercase. Default is FALSE.}
}
\value{
Invisibly returns a list containing summary metadata:
\itemize{
\item \code{n_observations}: total records processed
\item \code{n_included}: records included in shock computation
\item \code{n_excluded}: records excluded by configuration
\item \code{output_path}: normalized path to output HAR file
}
}
\description{
Computes compounded GEMPACK-style percentage shocks between initial and target values,
producing multi-period shock series for dynamic simulation models. The function
automatically aligns dimensions across datasets and exports results in HAR format.
}
\details{
\itemize{
\item Computes percentage shocks using compounded "power of tax" formula
\item Supports multiple periods defined via \code{timeline} configuration
\item Compatible with HAR, SL4, CSV, or XLSX input formats
\item Excludes self-trade or specified region-sector pairs when configured
\item Exports results as multi-header HAR file (one header per timeline period)
}
}
\examples{
# Example 1: Target-Based Shock Calculation
har_path <- system.file("extdata", "baserate.har", package = "HARplus")

# Sorting Column
mapping <- list(
  REG = c("USA", "EU", "ROW")
)

# Initial File
initial <- create_initial_config(
  path   = har_path,
  format = "har",
  header = "rTMS"
)

# Target File
target <- create_target_config(
  path   = har_path,
  type   = "har",
  header = "rTMS"
)

# Calculation Setup with Column Mapping
calc <- create_calc_config(
  column_mapping      = c(TRAD_COMM = "TRAD_COMM", REG = "REG", REG.1 = "REG.1"),
  timeline            = "1-5",
  exclude_self_trade  = TRUE
)

# Compute Shock Based on Initial and Target Values
shock_calculate(
  initial_config = initial,
  target_config  = target,
  calc_config    = calc,
  output_path    = file.path(tempdir(), "output_target.har"),
  dim_order      = mapping
)
}
\seealso{
\code{\link{shock_calculate_uniform}}, \code{\link{create_initial_config}},
\code{\link{create_target_config}}, \code{\link{create_calc_config}}, \code{\link{save_har}}
}
\author{
Pattawee Puangchit
}
