% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tidy.rms.R
\name{tidy.rms}
\alias{tidy.rms}
\title{Tidy a(n) rms model object}
\usage{
\method{tidy}{rms}(
  x,
  conf.int = FALSE,
  conf.level = 0.95,
  exponentiate = FALSE,
  ...,
  .add_print_p_and_stat_values = getOption("Greg.tidy_add_p_and_stat_values", default =
    FALSE)
)
}
\arguments{
\item{x}{An rms model, e.g. [`rms::cph()`], [`rms::lrm()`]}

\item{conf.int}{Logical indicating whether or not to include a confidence
interval in the tidied output. Defaults to \code{FALSE}.}

\item{conf.level}{The confidence level to use for the confidence interval
if \code{conf.int = TRUE}. Must be strictly greater than 0 and less than 1.
Defaults to 0.95, which corresponds to a 95 percent confidence interval.}

\item{exponentiate}{Logical indicating whether or not to exponentiate the
the coefficient estimates. This is typical for logistic and multinomial
regressions, but a bad idea if there is no log or logit link. Defaults
to \code{FALSE}.}

\item{...}{Additional arguments. Not used. Needed to match generic
signature only. \strong{Cautionary note:} Misspelled arguments will be
absorbed in \code{...}, where they will be ignored. If the misspelled
argument has a default value, the default value will be used.
For example, if you pass \code{conf.lvel = 0.9}, all computation will
proceed using \code{conf.level = 0.95}. Two exceptions here are:
\itemize{
\item \code{tidy()} methods will warn when supplied an \code{exponentiate} argument if
it will be ignored.
\item \code{augment()} methods will warn when supplied a \code{newdata} argument if it
will be ignored.
}}

\item{.add_print_p_and_stat_values}{For estimating print values there is a workaround that
relies on capturing output from the `print(x)` and is not considered safe.}
}
\value{
A tibble::tibble() with columns:
- `term` The name of the regression term.
- `factor` The factor if the term is a character/factor term.
- `column_term` The full name as in the original input data
- `estimate`	The estimated value of the regression term.
- `conf.high`	Upper bound on the confidence interval for the estimate.c
- `conf.low`	Lower bound on the confidence interval for the estimate.
- `p.value` The two-sided p-value associated with the observed statistic.
- `statistic` The value of a statistic to use in a hypothesis that the regression term is non-zero.
- `std.error` The standard error of the regression term.
}
\description{
Tidy summarizes information about the components of a model. A model component 
might be a single term in a regressions. Exactly what tidy considers to be a model 
component varies across models but is usually self-evident. If a model has several 
distinct types of components, you will need to specify which components to return.
}
\details{
This is a quick fix for addressing the lack of `rms`-compatibility with the `broom`
package, see [broom issue 30](https://github.com/tidymodels/broom/issues/30).
}
\examples{
library(rms)
library(broom)
library(tidyverse)

set.seed(10)
cov <- tibble(x1 = runif(200)) |> 
  mutate(x_bool_fact = if_else(x1 > 0.5,
                               "Yes",
                               sample(c("Yes", "No"), size = n(), replace = TRUE)),
         x_multi_fact = sample(c("Strange", "Factor", "Names"), size = n(), replace = TRUE),
         ftime = rexp(n()),
         fstatus = sample(0:1, size = n(), replace = TRUE),
         x_good_predictor = fstatus * runif(n()))

ddist <- datadist(cov)
options(datadist = "ddist")

cph_fit <- cph(Surv(ftime, fstatus) ~ x1 + x_bool_fact + 
                 x_multi_fact + x_good_predictor, data = cov)
tidy(cph_fit)
}
