#' Calculate Geometric Means from Count Tables
#'
#' This function computes the geometric mean for each sample in the given count table.
#' Users can choose how to handle missing values.
#'
#' @param count_table A data frame of gene count data (genes as rows, samples as columns). All columns must be numeric.
#' @param na.action Character: how to handle NAs. Options are
#'   * "omit" (default): ignore NAs and compute the geometric mean with remaining values
#'   * "fail": return NA for the sample if any NA is present
#' @return A data frame with the geometric means per sample and the sample IDs.
#' @examples
#' count_table <- data.frame(
#'   sample1 = c(1, 10, 100),
#'   sample2 = c(2, 20, 200),
#'   sample3 = c(3, 30, NA)
#' )
#' rownames(count_table) <- c("gene1", "gene2", "gene3")
#'
#' geomean(count_table)
#'
#' geomean(count_table, na.action = "fail")
#' @export
geomean <- function(count_table, na.action = c("omit", "fail")) {

  na.action <- match.arg(na.action)

  # Stop if input is not a data frame
  if (!is.data.frame(count_table)) {
    stop("count_table should be a data frame.")
  }

  geometric_mean <- function(x) {
    if (na.action == "fail" && any(is.na(x))) {
      return(NA_real_)
    }
    # Only use positive, non-NA values
    exp(mean(log(x[!is.na(x) & x > 0])))
  }

  geomeans <- apply(count_table, 2, geometric_mean)

  score_frame <- data.frame(GeoMean = geomeans)
  score_frame$ID <- colnames(count_table)

  return(score_frame)
}
