% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/postregression_functions.R
\name{AME}
\alias{AME}
\alias{manual_AME}
\alias{ACE}
\alias{AMIE}
\title{Calculate marginal effects}
\usage{
AME(
  object,
  baseline = NULL,
  vcov = TRUE,
  design = NULL,
  ignore_restrictions = FALSE,
  vcov.type = NULL,
  average_position = TRUE,
  verbose = TRUE,
  plot = TRUE
)

manual_AME(
  object,
  baseline,
  vcov = TRUE,
  design = NULL,
  extra_restriction = NULL,
  ignore_restrictions = FALSE,
  vcov.type = NULL,
  average_position = TRUE,
  verbose = TRUE,
  plot = TRUE
)

ACE(
  object,
  baseline,
  design = NULL,
  average_position = TRUE,
  ignore_restrictions = FALSE,
  extra_restriction = NULL,
  verbose = TRUE,
  plot = TRUE
)

AMIE(
  object,
  design = NULL,
  baseline = NULL,
  average_position = TRUE,
  ignore_restrictions = FALSE,
  verbose = FALSE,
  plot = TRUE
)
}
\arguments{
\item{object}{An object from \code{\link{FactorHet}} or \code{\link{FactorHet_mbo}}.}

\item{baseline}{A named list consisting of each factor and a corresponding
baseline level. The default (\code{NULL}) computes the effect for all
factors using the first level as the baseline. \code{NA} uses no baseline
and approximates the "marginal means" from Leeper et al. (2020).}

\item{vcov}{A logical value indicating whether the standard errors for the
AME should be computed. The default is \code{TRUE}. Standard errors are not
implemented for the AMIE.}

\item{design}{A dataset used to estimate the marginal effects. The default,
\code{NULL}, uses the estimation data.}

\item{ignore_restrictions}{A logical value about whether to ignore
randomization restrictions when calculating the marginal effects. The
default is \code{FALSE}. "Details" provides more information.}

\item{vcov.type}{A string indicating the type of standard errors to be
computed. The default is \code{NULL} and uses the default settings in
\code{\link{vcov.FactorHet}}; options are specified by that function's
\code{se.method} argument.}

\item{average_position}{A logical value indicating whether, for forced choice
designs, the "left" and "right" profiles should be averaged. The default is
\code{TRUE}. Goplerud et al. (2025) provide discussion of this point.}

\item{verbose}{A logical value as to whether more information should be
provided on the progress of estimating the effects. The default is
\code{TRUE}.}

\item{plot}{A logical value as to whether the function should print the plot
immediately or quietly provide an object containing the plot and data. The
default is \code{TRUE}.}

\item{extra_restriction}{A list of additional restrictions to include when
computing the marginal effects. The default is \code{NULL}, i.e. no
additional restrictions. "Details" provides more information about the use
of this function.}
}
\value{
Returns a named list with the underlying data (\code{"data"}) and the
  plot (\code{"plot"}).
}
\description{
Calculate the average marginal (component) effect (AME or AMCE), the average
combination effect (ACE), or the average marginal interaction effect (AMIE)
with a FactorHet model.
}
\details{
\bold{Choice of Baseline}: For ACE and AMIE, a choice of baseline is
required. See Egami and Imai (2019) for details. For AME, a choice of
baseline corresponds to a "standard" AME (see Egami and Imai 2019). The
option \code{NULL} chooses the first level of each factor. It can be manually
specified using a named list. If a named list is provided, only AMEs for
those named factors are calculated. This can be helpful if there are many
factors. 

If \code{NA} is provided as the baseline level, the AME is calculated without
a baseline; while this does not correspond to a "proper" AME, it is designed
to approximate the "marginal means" discussed in Leeper et al. (2020). Note
that in the presence of randomization restrictions, the quantity estimated
with a \code{NA} baseline may not be centered around 0.5. Ignoring the
randomization restrictions may be useful in this scenario. Supporting information
from Goplerud et al. (2025) provides more discussion of this point.

\bold{Randomization Restrictions}: Randomization restrictions can be set in
one of two ways. By default, FactorHet checks whether for each pairwise
combinations of factors, some combination of levels do not occur at all (e.g.
"doctor" and "high school") or whether some included interactions are
extremely rare (see \code{rare_threshold} in \code{\link{FactorHet_control}}). Those
are assumed to be the randomization restrictions implied by the design.
Setting \code{rare_threshold = 0} forces the inclusion of all interaction
terms.

If this procedure for automatically generating randomization restrictions is
inappropriate for a specific dataset, randomization restrictions can be set
manually as follows using the \code{manual_AME} function. First, set
\code{ignore_restrictions = TRUE}. This will ignore all "data-driven"
estimates of randomization restrictions. Second, the argument
\code{extra_restriction} should be a named list where the name of each
element corresponds to a factor (e.g. "Job") and each element is a vector of
the levels that \emph{cannot} be used. When using this approach, \code{AME}
should be used only for one factor at a time. An example is shown below.

\bold{Plots}: Note that for the ggplot2 visualizations of the ACE and AMIE,
gray squares indicate combinations that are excluded due to randomization
restrictions. White indicates baseline levels.
}
\examples{
data(immigration)
# Induce "fake" randomization restriction
immigration$joint_id <- paste(immigration$CaseID, immigration$contest_no)
remove_profiles <- subset(immigration, Plans == 'No plans' & Ed == 'GradDeg')
immigration <- subset(immigration, !(joint_id \%in\% remove_profiles$joint_id))
# Fit with one group and limited regularization for example only
fit <- FactorHet(Chosen_Immigrant ~ Plans + Ed + Country,
  design = immigration, lambda = 1e-4,
  K = 1, group = ~ CaseID, task = ~ contest_no, choice_order = ~ choice_id)
# Estimate AME of "Ed" with randomization restriction
est_AME <- AME(fit, baseline = list('Ed' = 'GradDeg'))
\donttest{
# Estimate AME ignoring randomization restriction
est_AME_norr <- AME(fit, 
  baseline = list('Ed' = 'GradDeg'), ignore_restrictions = TRUE)
# Estimate AME by manually specifying randomization restrictions
# this uses the 'manual_AME' function
est_AME_rr_manual <- manual_AME(fit,
  baseline = list('Ed' = 'GradDeg'), ignore_restrictions = TRUE,
  extra_restriction = list('Plans' = 'No plans'))
stopifnot(isTRUE(all.equal(est_AME$data, est_AME_rr_manual$data)))
# Estimate without baseline
est_MM <- AME(fit, baseline = list('Ed' = NA))
}
# Estimate ACE and AMIE
\donttest{
est_ACE <- ACE(fit, baseline = list('Ed' = 'GradDeg', 'Plans' = 'Has contract'))
}
est_AMIE <- AMIE(fit, baseline = list('Ed' = 'GradDeg', 'Plans' = 'Has contract'))
}
\references{
Egami, Naoki and Kosuke Imai. 2019. "Causal Interaction in
  Factorial Experiments: Application to Conjoint Analysis." \emph{Journal of the
  American Statistical Association}. 114(526):529-540.

  Goplerud, Max, Kosuke Imai, and Nicole E. Pashley. 2025. "Estimating
  Heterogeneous Causal Effects of High-Dimensional Treatments: Application to
  Conjoint Analysis." arxiv preprint: \url{https://arxiv.org/abs/2201.01357}
 
  Leeper, Thomas J., Sara B. Hobolt, and James Tilley. 2020. "Measuring Subgroup
  Preferences in Conjoint Experiments." \emph{Political Analysis}. 28(2):207-221.
}
