

#' @title Cost Matrix of \eqn{\ell^p}{l^p}-Form.
#' @description Compute cost matrices of \eqn{\ell^p}-form.
#' @param x matrix of size \eqn{n \times d} containing vectors \eqn{x_1, \ldots, x_n \in \mathbb{R}^d} (row-wise).
#' @param y matrix of size \eqn{m \times d} containing vectors \eqn{y_1, \ldots, y_m \in \mathbb{R}^d} (row-wise); `y = NULL` means that \eqn{y_i = x_i}.
#' @param p number \eqn{p \in (0, \infty]}.
#' @param q number \eqn{q \in (0, \infty)}.
#' @returns A \eqn{n \times m} matrix with entry at \eqn{i, j} being equal to
#' \deqn{
#'  \lVert x_i - y_j \rVert_p^q = \left[ \sum_{k=1}^d \lvert x_{i,k} - y_{j, k} \rvert^p \right]^{q/p}
#' }
#' For `p = Inf`, this is to be understood as the maximum norm to the power of \eqn{q}.
#' @examples
#' n <- 3
#' m <- 4
#' d <- 5
#' x <- runif(n * d) |> matrix(n, d)
#' y <- runif(m * d) |> matrix(m, d)
#' costm <- cost_matrix_lp(x, y)
#' print(costm)
#' @export
cost_matrix_lp <- \(x, y = NULL, p = 2, q = 1) {

    x <- as.matrix(x)
    stopifnot(
        is_num_mat(x),
        is_num_scalar(p),
        is_num_scalar(q),
        p > 0, q > 0
    )

    if (is.null(y)) {
        costMatrixSymmetricLp(x, p, q)
    }
    else {
        y <- as.matrix(y)
        stopifnot(is_num_mat(y), ncol(x) == ncol(y))
        costMatrixLp(x, y, p, q)
    }
}
