% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.maxlogL.R
\name{plot.maxlogL}
\alias{plot.maxlogL}
\title{Plot Residual Diagnostics for an \code{maxlogL} Object}
\usage{
\method{plot}{maxlogL}(
  x,
  type = c("rqres", "cox-snell", "martingale", "right-censored-deviance"),
  parameter = NULL,
  which.plots = NULL,
  caption = NULL,
  xvar = NULL,
  ...
)
}
\arguments{
\item{x}{a \code{maxlogL} object.}

\item{type}{a character with the type of residuals to be plotted.
The default value is \code{type = "rqres"}, which is used to
compute the normalized randomized quantile residuals.}

\item{parameter}{which parameter fitted values are required for
\code{type = "rqres"}. The default is the first one
defined in the pdf,e.g, in \code{dnorm}, the default
parameter is \code{mean}.}

\item{which.plots}{a subset of numbers to specify the plots. See details
for further information.}

\item{caption}{title of the plots. If \code{caption = NULL}, the default
values are used.}

\item{xvar}{an explanatory variable to plot the residuals against.}

\item{...}{further parameters for the \link{plot} method.}
}
\value{
Returns specified plots related to the residuals of the fitted
\code{maxlogL} model.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Provides plots of Cox-Snell, martingale Randomized quantile residuals.
}
\details{
If \code{type = "rqres"}, the available subset is \code{1:4}, referring to:
\itemize{
\item 1. Quantile residuals vs. fitted values (Tukey-Ascomb plot)
\item 2. Quantile residuals vs. index
\item 3. Density plot of quantile residuals
\item 4. Normal Q-Q plot of the quantile residuals.
}
}
\examples{
library(EstimationTools)

#----------------------------------------------------------------------------
# Example 1: Quantile residuals for a normal model
n <- 1000
x <- runif(n = n, -5, 6)
y <- rnorm(n = n, mean = -2 + 3 * x, sd = exp(1 + 0.3* x))
norm_data <- data.frame(y = y, x = x)

# It does not matter the order of distribution parameters
formulas <- list(sd.fo = ~ x, mean.fo = ~ x)
support <- list(interval = c(-Inf, Inf), type = 'continuous')

norm_mod <- maxlogLreg(formulas, y_dist = y ~ dnorm, support = support,
                       data = norm_data,
                       link = list(over = "sd", fun = "log_link"))

# Quantile residuals diagnostic plot
plot(norm_mod, type = "rqres")
plot(norm_mod, type = "rqres", parameter = "sd")

# Exclude Q-Q plot
plot(norm_mod, type = "rqres", which.plots = 1:3)


#----------------------------------------------------------------------------
# Example 2: Cox-Snell residuals for an exponential model
data(ALL_colosimo_table_4_1)
formulas <- list(scale.fo = ~ lwbc)
support <- list(interval = c(0, Inf), type = 'continuous')

ALL_exp_model <- maxlogLreg(
  formulas,
  fixed = list(shape = 1),
  y_dist = Surv(times, status) ~ dweibull,
  data = ALL_colosimo_table_4_1,
  support = support,
  link = list(over = "scale", fun = "log_link")
)

summary(ALL_exp_model)
plot(ALL_exp_model, type = "cox-snell")
plot(ALL_exp_model, type = "right-censored-deviance")

plot(ALL_exp_model, type = "martingale", xvar = NULL)
plot(ALL_exp_model, type = "martingale", xvar = "lwbc")


#----------------------------------------------------------------------------

}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
