#' SimMixDist function
#'
#' SimMixDist is a support function for generating samples from mixture distribution.
#' The main purpose of this function is to generate samples from non-normal distribution.
#'
#'@param nInv is a number of samples the function will generate.
#'@param mean is a mean of a normal distribution part of mixture distribution.
#'@param std is a standard deviation of a normal distribution part of mixture distribution.
#'@param p1 is a ratio of a normal distribution within a mixture distribution.
#'@param p2 is a ratio of a Cauchy distribution within a mixture distribution.
#'
#'@return This function returns a list of samples \code{V} generated by a mixture distribution.
#'
#'
#'@examples
#' # Generate simulation data with 100 samples with a mixture distribution
#' # The distribution consist of the following distributions:
#' # 1) 10% of uniform distribution range [-400,400];
#' # 2) 50% of normal distribution with mean = 40 and std =8; and
#' # 3) 40% of Cauchy distribution with location= 45 and scale = 2.
#'
#' V<-SimMixDist(nInv=100,mean=40,std=8,p1=0.1,p2=0.5)
#'
#'
#'@export
#'
SimMixDist<-function(nInv,mean,std,p1,p2)
{
  p3<-1-p1-p2
  X<-runif(100, min = 0, max = 99)
  rC1dist<-distr::r(UnivarMixingDistribution(Unif( -400,  400),
                                      Norm(mean=mean, sd=std*2), Cauchy(location=mean+5, scale=2),
                                      mixCoeff=c(p1, p2, p3)) )

  V<-c(rC1dist(nInv))
  return(list("V"=V))
}
#' SimNonNormalDist function
#'
#' SimNonNormalDist is a support function for generating samples from mixture distribution.
#' There are five categories. Each categories has \code{nInv} samples.
#' Categories C1,C2,C3, and C4 are dominated by C5 but none of them dominate each other.
#'
#' The main purpose of this function is to generate samples that contains domination relation among categories.
#'
#'@param nInv is a number of samples the function will generate for each category.
#'@param noisePer is ratio of uniform distribution within a mixture distribution.
#'It is considered as a uniform noise that make an approach to hardly distinguish whether one distribution dominates another.
#'
#'@return This function returns a list of samples \code{Values} and their category \code{Group} generated by a mixture distribution.
#'
#'\item{Values}{ A vector of samples generated by a mixture distribution. }
#'\item{Group}{ A list of categories associated with \code{Values}. }
#'\item{V1,...,V5}{Lists of sample vectors separated by categories.}
#'
#'
#' @examples
#' # Generate simulation data with 100 samples per categories with 10% of uniform noise
#'
#' simData<-SimNonNormalDist(nInv=100,noisePer=0.1)
#'
#'
#'@export
#'
SimNonNormalDist<-function(nInv,noisePer)
{
  if(missing(nInv)) {
    nInv = 100
    noisePer = 0.05
  }
  if(missing(noisePer)) {
    noisePer = 0.05
  }

  initMean=20
  stepMean=60
  std=8

  p1<-noisePer
  p2<-0.5
  p3<-1-p1-p2

  M1<-initMean + stepMean

  V1<-SimMixDist(nInv,M1,std,p1,p2)$V

  M1<-initMean+ stepMean
  V2<-SimMixDist(nInv,M1,std,p1,p2)$V

  M1<-initMean+ stepMean
  V3<-SimMixDist(nInv,M1,std,p1,p2)$V

  M1<-initMean+ stepMean
  V4<-SimMixDist(nInv,M1,std,p1,p2)$V

  M1<-initMean+2*stepMean
  V5<-SimMixDist(nInv,M1,std,p1,p2)$V

  simData<-c()
  simData$Group<-c(rep(c("C1"),times=nInv),rep(c("C2"),times=nInv),rep(c("C3"),times=nInv),rep(c("C4"),times=nInv),rep(c("C5"),times=nInv) )
  simData$Values <- c(V1,V2,V3,V4,V5)
  return(list("Values"=simData$Values,"Group"=simData$Group,"V1"=V1,"V2"=V2,"V3"=V3,"V4",V4,"V5"=V5))
}
#' checkSim3Res function
#'
#' checkSim3Res is a support function for checking whether an adjacency matrix of inferred
#' a dominant-distribution network \code{adjMat} is corrected w.r.t. generator SimNonNormalDist().
#'
#'@param adjMat is an adjacency matrix of inferred a dominant-distribution network.
#'@param flag is a flag of matrix. It should be set only to shift the low of matrix for comparison.
#'
#'@return This function returns precision, recall, and F1-score of inferred adjacency matrix.
#'
#' @examples
#' # Generate simulation data with 100 samples per categories
#'
#' simData<-SimNonNormalDist(nInv=100)
#'
#' # Performing ordering infernce from simData
#'
#' resultObj<-EDOIF(simData$Values,simData$Group)
#'
#' # Compare the inferred adjacency matrix with the ground truth
#'
#' checkSim3Res(adjMat=resultObj$adjMat)
#'
#'@export
#'
checkSim3Res<-function(adjMat,flag=0)
{
  TP<-0
  FP<-0
  FN<-0
  TN<-0
  prec<-0
  rec<-0
  F1<-0

  if(flag==0)
  {
    f=0
  }
  else
  {
    f=1
  }

  for(i in seq(2-f,4-f))
  {
    for(j in seq(1,i-1+f))
    {
      if(adjMat[i,j]==0)
        TN=TN+1
      else
        FP=FP+1

    }
  }
  for(j in seq(1,4))
  {
    if(adjMat[5-f,j]==1)
      TP=TP+1
    else
      FN=FN+1

  }
  prec<-TP/(TP+FP)
  rec<-TP/(TP+FN)
  F1<- 2*(prec*rec)/(prec+rec)

  if(TP+FP ==0)
    prec=0
  if(TP+FN ==0)
    rec=0
  if(prec+rec==0)
    F1=0
  return(list("F1"=F1,"prec"=prec,"rec"=rec))
}
