#' Generate Random Variables from a Truncated Gamma Distribution.
#'
#' Simulates random variables from a truncated gamma distribution, where the gamma distribution is bounded between user-specified lower and upper limits. This function is typically used to define priors (e.g., for the separation time in delayed treatment effect models).
#' @importFrom truncdist rtrunc
#' @param n Integer. Number of random variables to generate.
#' @param L Numeric. Lower bound of the truncation interval. Must satisfy \code{L <= U}.
#' @param U Numeric. Upper bound of the truncation interval. Must satisfy \code{U >= L}.
#' @param shape Numeric. Shape parameter of the underlying gamma distribution.
#' @param scale Numeric. Scale parameter of the underlying gamma distribution.
#' @return A numeric vector of length \code{n}, containing random variables drawn from a gamma distribution truncated to the interval \code{[L, U]}.
#' @examples
#' rtrunc_gamma(10, 1, 1, 1, 1)
#' @export
rtrunc_gamma <- function(n, L, U, shape, scale) {
  if (L == U) {

    return(rep(L, n))
  } else {

    return(rtrunc(n, spec ='gamma', a = L, b = U, shape = shape, rate = 1/scale))
  }
}
