#' @title Creates xpose database from Certara.RsNLME output files
#'
#' @description Imports results of an NLME run into xpose database
#' Use to import NLME model output files into \code{xpdb} object that is compatible
#' with existing model diagnostic function in \code{Xpose} package.
#'
#' @param dir Path to NLME Run directory. Current working directory is used if \code{dir} not given.
#' @param modelName name of the model to be written in \code{xpdb$summary$value} with run label
#' @param dmpFile NLME generated output file.
#' @param dmp.txt NLME generated output from dmpFile (substitutes dmpFile if presented).
#' @param dataFile Input file for NLME Run.
#' @param logFile engine log file
#' @param ConvergenceData optional data frame with Nlme convergence info.
#' @param progresstxt optional NLME-generated file 'progress.txt' with convergence info.
#' \code{ConvergenceData} has more priority if both are given.
#'
#' @details Not all functionality from the \code{xpose} package is supported.
#'
#' @return \code{xpdb} object
#'
#' @examples
#' \donttest{
#' # files in arguments supposed to be in the current working directory:
#' xp <- xposeNlme(
#'   dir = getwd(),
#'   modelName = "PMLModel",
#'   dmpFile = "dmp.txt",
#'   dataFile = "data1.txt",
#'   logFile = "nlme7engine.log",
#'   progresstxt = "progress.txt"
#' )
#'
#' # using dmp.txt structure and Convergence Data loaded previously:
#' xp <- xposeNlme(
#'   dir = "~/Model1/",
#'   modelName = "Model1",
#'   dmp.txt = dmp.txt,
#'   dataFile = "Data.csv",
#'   logFile = "nlme7engine.log",
#'   ConvergenceData = ConvergenceData
#' )
#'
#' # explore unique covariate plots specific to Certara.Xpose.NLME:
#' nlme.cov.splom(xp, covColNames = c("AGE", "WT"))
#' nlme.par.vs.cov(xp, covColNames = c("AGE", "WT"))
#'
#' res_vs_cov(xp, covariate = "AGE", res = "IWRES")
#'
#' # or use existing plotting functions from the xpose package
#' library(xpose)
#' dv_vs_pred(xp)
#' res_vs_idv(xp)
#' }
#'
#' @export
xposeNlme <- function(dir = "",
                      modelName = "",
                      dmpFile = "dmp.txt",
                      dmp.txt = NULL,
                      dataFile = "data1.txt",
                      logFile = "nlme7engine.log",
                      ConvergenceData = NULL,
                      progresstxt = "progress.txt") {
  # prepare files
  FilesToCheck <- c(dataFile = dataFile,
                    logFile = logFile)

  if (is.null(dmp.txt)) {
    FilesToCheck <- c(FilesToCheck,
                      dmpFile = dmpFile)
  }

  for (NLMEFileIndex in seq_along(FilesToCheck)) {
    NLMEFile <- .check_filePath(FilesToCheck[NLMEFileIndex], dir)

    assign(names(FilesToCheck[NLMEFileIndex]), NLMEFile)
  }

  if (is.null(dmp.txt)) {
    source(dmpFile, local = TRUE)
  }

  nlme7engineLines <- readLines(logFile)
  InputUnits <- read_dataFile(dataFile)
  listInput <- create_input(InputUnits, dmp.txt$cols1.txt)

  xp <-
    create_xposeNlme(modelName = modelName,
                     dmp.txt = dmp.txt,
                     listInput = listInput,
                     nlme7engineLines = nlme7engineLines,
                     ConvergenceData = ConvergenceData,
                     dir = dir,
                     progresstxt = progresstxt,
                     Sort = NULL)

  xp
}
