CustomSpace <- function(PMLCode = character(),
                        SpaceName = character(),
                        TimeBased = logical(),
                        Responses = list(),
                        CustomCovariates = list(),
                        CustomDosepoints = list(),
                        CustomStParms = list(),
                        CustomFixefs = list(),
                        CustomRanefs = list(),
                        CFs = list(),
                        CustomDerivs = list(),
                        Transits = list(),
                        CustomUrines = list()) {
  ArgsList <- as.list(environment())
  stopifnot(is.character(PMLCode))
  stopifnot(is.logical(TimeBased))
  stopifnot(is.character(SpaceName))

  ModelHeaderPattern <- "^\\W*\\w+\\(\\)\\W*\\{\\n*"
  if (length(PMLCode) > 0 && grepl(ModelHeaderPattern, PMLCode)) {
    # need to remove the header
    PMLCode <- gsub(ModelHeaderPattern, "", PMLCode)
    PMLCode <- gsub("\\n*\\}\\W*$", "", PMLCode)
  }

  for (i in 4:length(ArgsList)) {
    if (!is.list(ArgsList[[i]])) {
      stop(names(ArgsList)[i], " should be a list.")
    }
  }

  if (!nchar(PMLCode) &&
      !nchar(SpaceName)) {
    SpaceName <-
      "EmptySpace"
  }

  structure(
    list(
      Type = "Custom",
      PMLCode = PMLCode,
      TimeBased = TimeBased,
      Responses = Responses,
      CustomCovariates = CustomCovariates,
      CustomDosepoints = CustomDosepoints,
      CustomStParms = CustomStParms,
      CustomFixefs = CustomFixefs,
      CustomRanefs = CustomRanefs,
      CFs = CFs,
      CustomDerivs = CustomDerivs,
      Transits = Transits,
      CustomUrines = CustomUrines,
      SpaceName = SpaceName
    ),
    class = "CustomSpace"
  )
}

#' Output a Custom Space
#'
#' This function generates the PML code representation of a custom space.
#'
#' @param x A `CustomSpace` object.
#' @param ... Additional arguments (not used).
#'
#' @return A character string containing the PML code.
#'
#' @export
output.CustomSpace <- function(x, ...) {
  paste0(gsub("\\\n(?!=\\\t)", "\\\n\\\t", x$PMLCode, perl = TRUE), collapse = "\n")
}

#' @export
print.CustomSpace <- function(x, ...) {
  on.exit(clean_TokensEnv(e = TokensEnv))
  cat(output(x))
}

#' Add a Custom Space to a PMLModels Object
#'
#' Adds a new model space, defined by custom PML code, to an existing collection
#' of model spaces (a `PMLModels` object).
#'
#' @param Spaces A `PMLModels` object (a named list) representing the existing
#'   collection of model spaces to which the new custom space will be added.
#'   This can be an empty list or a previously created `PMLModels` object.
#' @inheritParams create_CustomSpace
#'
#' @return An updated `PMLModels` object (a named list) containing all the
#'   original spaces plus the newly added custom space.
#'
#' @details
#' This function serves as a wrapper around [create_CustomSpace()]. It first
#' calls `create_CustomSpace` using the provided `CustomCode` and `SpaceName`
#' to parse the code and create a representation of the new custom space.
#' The name of this new space is either the provided `SpaceName` or one
#' automatically generated by `create_CustomSpace` if `SpaceName` was omitted
#' or empty (e.g., `"l<number>"` based on code length).
#'
#' @seealso [create_CustomSpace()], [add_Spaces()], [create_ModelPK()],
#'   [create_ModelPD()]
#'
#' @examples
#' # Start with some built-in models
#' pk_models <- create_ModelPK(CompartmentsNumber = 1)
#'
#' # Define custom code
#' custom_pml <- "test() {
#'   cfMicro(A1, Cl / V)
#'   dosepoint(A1)
#'   C = A1 / V
#'   error(CEps = 1)
#'   observe(CObs = C + CEps)
#'   stparm(V = tvV * exp(nV))
#'   stparm(Cl = tvCl * exp(nCl))
#'   fixef(tvV = c(, 1, ))
#'   fixef(tvCl = c(, 1, ))
#'   ranef(block(nV, nCl) = c(1, 0.001, 1))
#' }
#' "
#'
#' # Add custom space with an explicit name
#' all_models <-
#'   add_CustomSpace(pk_models, custom_pml, SpaceName = "1cptOmegaBlock")
#' names(all_models)
#'
#' # Add another custom space with auto-generated name
#' all_models_2 <- add_CustomSpace(all_models,
#'   "test() {
#'   cfMicro(A1, Cl / V)
#'   dosepoint(A1)
#'   C = A1 / V
#'   error(CEps = 1)
#'   observe(CObs = C + C ^ (0.5) * CEps)
#'   stparm(V = tvV * exp(nV))
#'   stparm(Cl = tvCl * exp(nCl))
#'   fixef(tvV = c(, 1, ))
#'   fixef(tvCl = c(, 1, ))
#'   ranef(block(nV, nCl) = c(1, 0.001, 1))
#' }
#' ")
#' names(all_models_2) # Will include original names + "l<number>"
#'
#' @export
add_CustomSpace <- function(Spaces, CustomCode, SpaceName = character()) {
  stopifnot(is.list(Spaces))

  NewCustomSpace <- create_CustomSpace(CustomCode, SpaceName)
  SpaceName <- names(NewCustomSpace)
  NewCustomSpace <- NewCustomSpace[[1]]

  SpaceNameOriginal <- SpaceName
  while (SpaceName %in% names(Spaces)) {
    SpaceName <- paste0(SpaceName, "_")
  }

  if (SpaceName != SpaceNameOriginal
        && methods::hasArg(SpaceName)) {
    warning("SpaceName '", SpaceNameOriginal, "' already exists. ",
            "Renaming to '", SpaceName, "'")
  }

  add_Spaces(Spaces, list(NewCustomSpace), SpaceName)
}

#' Add Multiple Model Spaces to a PMLModels Object
#'
#' Merges a list of new model spaces into an existing `PMLModels` object.
#'
#' @param Spaces A `PMLModels` object (a named list) representing the existing
#'   collection of model spaces. Can be an empty list or a previously created
#'   `PMLModels` object.
#' @param NewSpaces A list where each element is an internal representation of a
#'   single model space (e.g., the list structure produced by `create_ModelPK`
#'   or `create_CustomSpace`). If this list is named, these
#'   names will be used unless overridden by `NewSpacesNames`.
#' @param NewSpacesNames An optional character vector providing explicit names for
#'   the spaces listed in `NewSpaces`. If provided:
#'   If omitted, the names attached to the `NewSpaces` list itself will be used.
#'
#' @return An updated `PMLModels` object (a named list) containing all the
#'   original spaces plus all spaces from `NewSpaces`. The class "PMLModels"
#'   is preserved.
#'
#' @details
#' This function provides a general mechanism for combining collections of model
#' spaces.
#'
#' Naming and Collision Handling:
#' This behavior differs from [add_CustomSpace()], which automatically renames
#' on collision. `add_Spaces` requires unique, non-colliding names for the merge.
#'
#' @seealso [add_CustomSpace()] for adding a single custom space with automatic renaming.
#'
#' @examples
#' pk_model1 <-
#'   create_ModelPK(CompartmentsNumber = 1,
#'                  Absorption = "Intravenous")
#' pk_models2 <- create_ModelPK(CompartmentsNumber = c(2, 3),
#'                              Absorption = "First-Order")
#'
#' # Combine two PMLModels objects (using names from pk_models2)
#' combined1 <- add_Spaces(pk_model1, pk_models2)
#' names(combined1)
#'
#' # Combine using explicit new names
#' combined2 <-
#'   add_Spaces(pk_model1,
#'              pk_models2,
#'              NewSpacesNames = c("Model_A", "Model_B"))
#' names(combined2)
#'
#' # Add a list containing a single custom space
#' custom_pml <- "test(){ fixef(p=1) }"
#' custom_space_list <-
#'   create_CustomSpace(custom_pml, "MyCustom")
#' combined3 <-
#'   add_Spaces(pk_model1, custom_space_list)
#' names(combined3)
#'
#' @export
add_Spaces <- function(Spaces, NewSpaces, NewSpacesNames = character()) {
  stopifnot(is.list(Spaces))
  if (length(Spaces) == 1 &&
      length(Spaces[[1]]$PMLCode) > 0 &&
      nchar(Spaces[[1]]$PMLCode) == 0) {
    # only empty is given
    Spaces <- list()
  } else {
    stopifnot(SpacesClass = inherits(Spaces, "PMLModels"))
  }

  if (methods::hasArg(NewSpacesNames)) {
    if (length(NewSpaces) != length(NewSpacesNames)) {
      stop("NewSpacesNames must be the same length as NewSpaces")
    }

    if (any(duplicated(NewSpacesNames))) {
      stop("NewSpacesNames must be unique")
    }

    if (any(NewSpacesNames %in% names(Spaces))) {
      stop("NewSpacesNames must not be in the existing spaces")
    }

    if (any(length(NewSpacesNames) == 0) ||
            any(NewSpacesNames == "")) {
      stop("NewSpacesNames must not be empty")
    }

    if (any(!is.character(NewSpacesNames))) {
      stop("NewSpacesNames must be character")
    }

    NewSpaces <-
      stats::setNames(NewSpaces, NewSpacesNames)
  } else {
    NewSpacesNames <- names(NewSpaces)
    if (any(length(NewSpacesNames) == 0) ||
            any(NewSpacesNames == "")) {
      warning("NewSpacesNames must not be empty", call. = FALSE)
    }
  }

  if (any(NewSpacesNames %in% names(Spaces))) {
    stop("The names of the newly spaces added must not be in the existing spaces names.\n",
         "The existing names:", paste(names(Spaces), collapse = ", "),
         "The new names:", paste(NewSpacesNames, collapse = ", "))
  }

  Spaces <-
    stats::setNames(c(Spaces, NewSpaces),
                    c(names(Spaces), NewSpacesNames))

  structure(Spaces,
            class = "PMLModels")
}
