#' Fit Censored Linear Regression Model under Scale Mixtures of Skew-Normal Distributions
#'
#' Fits a univariate linear regression model with censoring and/or missing values in the response variable, assuming it follows a distribution from the Scale Mixtures of Skew-Normal (SMSN) family. Computes standard errors using the empirical information matrix and provides model selection criteria (AIC, BIC, CAIC, HQ). Optionally generates envelope plots based on martingale residuals.
#'
#' @param cc Indicator vector for incomplete observations of length \code{n}. Each element should be \code{0} if the observation is fully observed, or \code{1} if it is incomplete (either censored or missing).
#' @param x Design matrix (of dimension \code{n x p}) corresponding to the covariates in the linear predictor.
#' @param y Response vector of length \code{n}. For fully observed data, it contains the observed values. In the case of right or left censoring, it represents the censoring limit. For interval censoring, it corresponds to the lower bound of the censoring interval. Missing values (\code{NA}) are allowed.
#' @param beta Optional initial values for the regression coefficients. Default is \code{NULL}.
#' @param sigma2 Optional initial value for the scale parameter. Default is \code{NULL}.
#' @param lambda Optional initial value for the shape parameter (for skewed distributions). Default is \code{NULL}.
#' @param nu Optional initial value for the distribution-specific parameter. Required for \code{T}, \code{ST}, \code{CN}, and \code{SCN} families. Must be a two-dimensional vector for \code{CN} and \code{SCN}. Should not be provided for \code{N} or \code{SN}. Default is \code{NULL}.
#' @param cens Character indicating the type of censoring. Should be one of \code{"Left"}, \code{"Right"} or \code{"Int"}. Default is \code{"Int"}.
#' @param UL Vector of upper limits of length \code{n} for interval-censored observations. Must be provided when \code{cens = "Int"}.
#' @param get.init Logical; if \code{TRUE}, initial values are automatically computed. If \code{FALSE}, initial values must be provided. Default is \code{TRUE}.
#' @param show.envelope Logical; if \code{TRUE}, an envelope plot based on transformed martingale residuals is produced. Default is \code{FALSE}.
#' @param error Convergence threshold for the algorithm. Default is \code{0.0001}.
#' @param iter.max Maximum number of iterations allowed in the algorithm. Default is \code{300}.
#' @param family Character string indicating the distribution family. Possible values include: \code{"SN"} (Skew-Normal), \code{"ST"} (Skew-t), \code{"SCN"} (Skew Contaminated Normal), \code{"N"} (Normal), \code{"T"} (Student-t), \code{"CN"} (Contaminated Normal). Default is \code{"ST"}.
#' @param verbose Logical indicating whether results should be printed to the console. Default is \code{TRUE}.
#'
#' @details
#' The model assumes that the response variable follows a distribution from the Scale Mixtures of Skew-Normal (SMSN) family, which allows for heavy tails and/or asymmetry.
#'
#' Interval censoring is a general framework that includes left and right censoring and missing responses, providing a unified treatment for all cases.
#'
#' For the Skew Contaminated Normal (\code{"SCN"}) and the Contaminated Normal (\code{"CN"}) distributions, the \code{nu} parameter must be a two-dimensional vector with values in the interval \eqn{(0, 1)}.
#'
#' @return A list with the following components:
#' \item{beta}{Estimated regression coefficients.}
#' \item{sigma2}{Estimated scale parameter.}
#' \item{lambda}{Estimated shape parameter. For symmetric distributions ("N", "T", "CN"), this is zero.}
#' \item{nu}{Estimated parameters of the scale mixture distribution. \code{NULL} for "SN" and "N" families. A scalar for "ST" and "T", and a vector for "SCN" and "CN".}
#' \item{SE}{Standard errors of the estimated parameters.}
#' \item{iter}{Number of iterations until convergence.}
#' \item{logver}{Value of the log-likelihood function at convergence, computed under the fitted model.}
#' \item{AIC, BIC, CAIC, HQ}{Information criteria for model selection.}
#' \item{residual}{Transformed martingale residuals used for envelope plots. Returned only if \code{show.envelope = TRUE}; otherwise \code{NULL}.}
#'
#' @references
#' Gil, Y. A., Garay, A. M. & Lachos, V. H.
#' Likelihood-based inference for interval censored regression models under heavy-tailed distributions.
#' *Stat Methods Appl* 34, 519–544 (2025). \doi{10.1007/s10260-025-00797-x}.
#'
#' @examples
#' # See examples in ?gen_SMSNCens_sample  for a complete workflow
#' # illustrating data generation and model fitting.
#'
#' @export
#' @importFrom stats coefficients dnorm dt integrate lm optim pnorm ppoints pt qnorm quantile rbeta rgamma rnorm runif sd rt
CensRegSMSN <- function(cc, x, y, beta = NULL, sigma2 = NULL, lambda = NULL, nu = NULL, cens="Int", UL=NULL, get.init = TRUE, show.envelope = FALSE, error = 0.0001, iter.max = 300, family = "ST", verbose = TRUE){

  if (ncol(as.matrix(y)) > 1)
    stop("Only univariate linear regression supported!")
  if (ncol(as.matrix(cc)) > 1)
    stop("Only univariate linear regression supported!")
  if (length(y) == 0)
    stop("Argument 'y' must not be empty.")
  if (!is.numeric(y))
    stop("The argument 'y' must be numeric.")
  if (length(y) != nrow(as.matrix(x)))
    stop("Incompatible dimensions: 'x' must have the same number of rows as 'y'.")
  if (!is.matrix(x) || !is.numeric(x))
    stop("The argument 'x' must be a numeric matrix.")
  if (anyNA(x))
    stop("Missing values are not allowed in 'x'.")
  if (length(cc) != length(y))
    stop("Incompatible dimensions: 'cc' must have the same length as 'y'.")
  if (!all(cc %in% c(0, 1)))
    stop("It is expected that 'cc' contains only 0 (observed) and 1 (censored and missing) values.")
  if ((family != "ST") && (family != "T") && (family != "SN") && (family != "N") && (family != "SCN") && (family != "CN"))
    stop(paste0("Family '", family, "' not recognized. ", "Valid options are: SN, ST, SCN, N, T, CN."))
  # Verificaciones según tipo de censura
  if (cens %in% c("Right", "Left")) {
    if (!is.null(UL))
      stop("For 'Right' or 'Left' censoring, the argument 'UL' must be NULL.")
    if (any(is.na(y)))
      stop("Missing values in 'y' are only allowed under 'Int' censoring.")
    if (any(is.infinite(y)))
      stop("Values '-Inf' or 'Inf' in 'y' are only allowed under 'Int' censoring.")
  } else if (cens == "Int") {
    if (is.null(UL))
      stop("For 'Int' censoring, the argument 'UL' must be provided.")
    if (length(UL) != length(y))
      stop("Incompatible dimensions: 'UL' must have the same length as 'y'.")
    if (any(cc == 1 & !is.na(y) & y >= UL))
      stop("For interval-censored observations (cc == 1), it must hold that y < UL.")
  } else {
    stop("Invalid censoring type. The argument 'cens' must be either 'Right', 'Left' or 'Int'.")
  }
  if (!is.logical(show.envelope) || length(show.envelope) != 1)
    stop("Argument 'show.envelope' must be a single logical value: TRUE or FALSE.")
  if (!is.numeric(error) || error <= 0)
    stop("'error' must be a small positive number.")
  if (!is.numeric(iter.max) || iter.max <= 0)
    stop("'iter.max' must be a positive integer.")


  p <- ncol(x)
  n <- nrow(x)

  if (get.init == FALSE){
    if (!is.numeric(beta) || length(beta) != p)
      stop("When 'get.init = FALSE', 'beta' must be a numeric vector with length equal to the number of columns in 'x'.")
    if (length(sigma2) == 0 || sigma2 <= 0)
      stop("The argument 'sigma2' must be a positive scalar.")
    if (length(lambda) == 0)
      stop("The argument 'lambda' must be provided.")
    if (!is.numeric(lambda) || length(lambda) != 1)
      stop("The argument 'lambda' must be a numeric scalar.")
    if (family %in% c("T", "ST", "CN", "SCN")) {
      if (length(nu) == 0)
        stop(sprintf("When 'family = \"%s\"', the argument 'nu' must be provided.", family))
      if (!is.numeric(nu))
        stop("The argument 'nu' must be numeric.")
    }
    if (family == "T" || family == "ST") {
      if (length(nu) > 1)
        stop("The argument 'nu' must be a scalar for the T or ST distribution.")
      if (nu <= 2)
        stop("The argument 'nu' must be greater than 2 for the T or ST distribution.")
    }
    if (family %in% c("CN", "SCN")) {
      if (length(nu) != 2)
        stop("The argument 'nu' must be a bidimensional vector for the CN or SCN distribution.")
      if (any(nu <= 0 | nu >= 1))
        stop("Both elements of 'nu' must lie in the interval (0, 1) for the CN or SCN distribution.")
    }

    shape  <- lambda
    delta  <- shape / (sqrt(1 + shape^2))
    Delta  <- sqrt(sigma2)*delta
    Gama   <- sigma2 - Delta^2
    muaux      <- x%*%beta
  }

  x        <- as.matrix(x)

  if (get.init == TRUE){
    y0        <- y[cc==0]
    x0        <- x[cc==0,]
    x0        <- as.matrix(x0)
    reg       <- lm(y0 ~ -1 + x0[, 1:p])
    # reg       <- lm(y0 ~ x0[ , 2:p])
    beta      <- as.vector(coefficients(reg), mode = "numeric")
    sigma2    <- sum((y0 - x0%*%beta)^2)/(n - p)

    if (family %in% c("N", "T", "CN")) {
      shape   <- 0
    } else {
      shape   <- as.numeric(sign(skewness_simp(y0 - x0 %*% beta)) * 3)
    }

    delta     <- shape / (sqrt(1 + shape^2))
    Delta     <- sqrt(sigma2)*delta
    Gama      <- sigma2 - Delta^2

    muaux     <- x%*%beta

    if (family %in% c("SCN", "CN")) {
      nu <- c(0.5, 0.5)
    } else if (family %in% c("ST", "T")) {
      nu <- 3
    } else {
      nu <- NULL
    }

  }

  Lim1  <- Lim2 <- vector(mode = "numeric", length = length(y))

  if (cens=="Left")
  {
    Lim1 <- rep(-Inf,n)
    Lim2 <- y
  }
  if (cens=="Right")
  {
    Lim1 <- y
    Lim2 <- rep(Inf,n)
  }
  if (cens=="Int")
  {
    cens <- "Interv"
    Lim1 <- y
    Lim2 <- UL

    idx_na       <- is.na(y)
    y[idx_na]    <- -Inf
    Lim1[idx_na] <- -Inf
    Lim2[idx_na] <-  Inf
    cc[idx_na]   <- 1
  }


  #########################
  ##   Skew-t
  #########################
   if(family=="ST"){

    cont       <- 0
    criterio   <- 1
    lkante     <- 1
    ychap      <- y
    n.par      <- p + 2

    y1         <- y[cc==1]
    Lim1aux    <- Lim1[cc==1]
    Lim2aux    <- Lim2[cc==1]

    k1         <- sqrt(nu/2)*gamma((nu-1)/2)/gamma(nu/2)
    b          <- -sqrt(2/pi)*k1
    mu         <- muaux + b*Delta

    while(criterio > error){

      cont     <- (cont + 1)
      # print(cont)
      # print(c(beta,sigma2,shape,nu))

      dj       <- ((y - mu)/sqrt(sigma2))^2
      Mtij2    <- 1/(1 + (Delta^2)*(Gama^(-1)))
      Mtij     <- sqrt(Mtij2)
      mutij    <- Mtij2*Delta*(Gama^(-1))*(y - mu)
      A        <- mutij / Mtij
      cnu      <- 2*gamma((nu+1)/2)/(gamma(nu/2)*sqrt(nu*pi*(1 + shape^2)))
      E        <- (2*(nu)^(nu/2)*gamma((2+nu)/2)*((dj + nu + A^2))^(-(2+nu)/2)) / (gamma(nu/2)*pi*sqrt(sigma2)*dt.ls(y, mu, sigma2, shape, nu))
      u        <- ((4*(nu)^(nu/2)*gamma((3+nu)/2)*(dj + nu)^(-(nu+3)/2)) / (gamma(nu/2)*sqrt(pi)*sqrt(sigma2)*dt.ls(y, mu, sigma2, shape, nu)) )*pt(sqrt((3+nu)/(dj+nu))*A,3+nu)

      S1       <- u
      S2       <- ((mutij+b)*u + Mtij*E)
      S3       <- ((mutij+b)^2*u + Mtij2 + Mtij*(mutij+2*b)*E)

      E00      <- S1
      E01      <- y*S1
      E02      <- y^2*S1
      E10      <- S2
      E20      <- S3
      E11      <- y*S2
      sigma2s  <- nu/(nu + 2)*sigma2
      sigma2ss <- nu/((nu + 1)*(1 + shape^2))*sigma2

      Aux1     <- pSNI2(Lim1,Lim2, mu,  sigma2s, shape, nu+2, type = "ST")
      Aux11    <- pSNI2(Lim1,Lim2, mu, sigma2ss,     0, nu+1, type = "ST")
      Aux2     <- pSNI2(Lim1,Lim2, mu,   sigma2, shape,   nu, type = "ST")

      if(length(which(Aux2 == 0)) > 0) Aux2[which(Aux2 == 0)] <- .Machine$double.xmin

      mu1      <- mu[cc==1]
      np       <- length(mu1)
      aux1MomW <- aux2MomS <- aux3MomS <- matrix(0, np, 2)

      if (np > 0) {
        for(j in 1:np){
          A1a          <- MomenSNI2023(mu1[j],  sigma2s, shape, nu+2, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "ST")
          A2a          <- MomenSNI2023(mu1[j], sigma2ss,     0, nu+1, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "ST")
          A3a          <- MomenSNI2023(mu1[j],   sigma2, shape,   nu, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "ST")
          aux1MomW[j,] <- c(A1a$EUY1, A1a$EUY2)
          aux2MomS[j,] <- c(A2a$EUY1, A2a$EUY2)
          aux3MomS[j,] <- c(A3a$EUY1, A3a$EUY2)
        }

        P1aux <- P2aux <- WW <- u
        P1aux[cc==1]   <- aux1MomW[,1]
        P2aux[cc==1]   <- aux1MomW[,2]
        WW[cc==1]      <- aux2MomS[,1]

        Wphi           <- Aux11/Aux2
        E00Aux         <- Aux1/Aux2
        E01Aux         <- Aux1/Aux2*P1aux
        E02Aux         <- Aux1/Aux2*P2aux
        E10Aux         <- Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + Mtij*cnu*Wphi + b*E00Aux
        E20Aux         <- (Delta/(Gama + Delta^2))^2*(E02Aux - 2*E01Aux*mu + mu^2*E00Aux) + 2*b*Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + b^2*E00Aux + Mtij2 + Mtij*((Delta/(Gama + Delta^2))*cnu*Wphi*(WW - mu) + 2*b*cnu*Wphi )

        E11Aux         <- Delta/(Gama + Delta^2)*(E02Aux - E01Aux*mu) + Mtij*Wphi*cnu*WW + b*E01Aux

        ychap[cc==1]   <- aux3MomS[,1]
        E00[cc==1]     <- E00Aux[cc==1]
        E01[cc==1]     <- E01Aux[cc==1]
        E02[cc==1]     <- E02Aux[cc==1]
        E10[cc==1]     <- E10Aux[cc==1]
        E20[cc==1]     <- E20Aux[cc==1]
        E11[cc==1]     <- E11Aux[cc==1]
      }

      beta           <- solve(t(x)%*%diag(c(E00))%*%x)%*%(t(x)%*%matrix(E01 - E10*Delta, n, 1))
      muaux          <- x%*%beta
      Delta          <- sum(E11 - E10*muaux)/sum(E20)
      Gama           <- sum(E02 - 2*E01*muaux + E00*muaux^2 + Delta^2*E20 - 2*Delta*E11 + 2*Delta*E10*muaux)/n
      sigma2         <- Gama + Delta^2
      shape          <- ((sigma2^(-1/2))*Delta)/(sqrt(1 - (Delta^2)*(sigma2^(-1))))

      k1             <- sqrt(nu/2)*gamma((nu-1)/2)/gamma(nu/2)
      b              <- -sqrt(2/pi)*k1
      mu             <- muaux + b*Delta

      f<-function(nu){
          sum(log(dSTMod(cc, y, mu, sigma2 , shape, nu, cens=cens, LS=Lim2)))
      }

      nu             <- optim(par = 0, fn = f, control = list(fnscale = -1), method = "L-BFGS-B", lower = 2.01, upper = 40)$par

      lk             <- f(nu)
      logver         <- lk
      criterio       <- abs((lk/lkante - 1))
      lkante         <- lk

      if (cont==iter.max){
        criterio     <- error/10
      }
    }

    teta_novo        <- matrix(c(beta, sigma2, shape, nu), ncol = 1) # to compute the number of parameters


    #########################
    ##  Information Matrix
    #########################

    sbeta            <- c()
    ssigma2          <- c()
    slambda          <- c()
    MIE              <- matrix(0, p+2, p+2)
    S                <- matrix(0,   1, p+2)
    sigma            <- sqrt(sigma2)
    lambda           <- shape

    for(i in 1:n){
      sbeta          <- ((1 + lambda^2)/sigma2)*(E01[i]*t(x[i,]) - E00[i]*t(x[i,])*muaux[i] - Delta*E10[i]*t(x[i,]))
      ssigma2        <- -1/(2*sigma2) + ((1 + lambda^2)/(2*sigma2^2))*(E02[i] - 2*E01[i]*muaux[i] + (t(muaux[i])%*%muaux[i])*E00[i]) - ((lambda*sqrt(1+lambda^2))/(2*sigma^3))*(E11[i] - E10[i]*muaux[i])
      slambda        <- lambda/(1 + lambda^2) - (lambda/sigma2)*(E02[i] - 2*E01[i]*muaux[i] + E00[i]*(t(muaux[i])%*%muaux[i])) + ((1 + 2*lambda^2)/(sigma*sqrt(1+lambda^2)))*(E11[i] - E10[i]*muaux[i]) - lambda*E20[i]
      S              <- c(sbeta, ssigma2, slambda)
      MIE1           <- S%*%t(S)
      ind            <- lower.tri(MIE1)
      MIE1[ind]      <- t(MIE1)[ind]
      MIE            <- MIE1 + MIE
    }

    se               <- sqrt(diag(solve(MIE)))

    resmt <- NULL
    if(show.envelope){
      envelop <- EnvelopeRMT_ggp(teta_novo, y, x, cc, type = "ST", family = "ST")
      print(envelop)
      resmt <- envelop$plot_env$res$resmt
    }
  }

  #########################
  ## Student-t
  #########################

  if(family=="T"){

    cont       <- 0
    criterio   <- 1
    lkante     <- 1
    shape      <- 0
    ychap      <- y
    n.par      <- p + 1

    y1         <- y[cc==1]
    Lim1aux    <- Lim1[cc==1]
    Lim2aux    <- Lim2[cc==1]

    mu         <- muaux

    while(criterio > error){

      cont     <- (cont+1)
      # print(cont)
      # print(c(beta,sigma2,shape,nu))

      dj       <- ((y - mu)/sqrt(sigma2))^2
      Mtij2    <- 1/(1 + (Delta^2)*(Gama^(-1)))
      Mtij     <- sqrt(Mtij2)
      mutij    <- Mtij2*Delta*(Gama^(-1))*(y - mu)
      A        <- mutij / Mtij
      cnu      <- 2*gamma((nu + 1)/2)/(gamma(nu/2)*sqrt(nu*pi*(1 + shape^2)))
      E        <- (2*(nu)^(nu/2)*gamma((2+nu)/2)*((dj + nu + A^2))^(-(2+nu)/2)) / (gamma(nu/2)*pi*sqrt(sigma2)*dt.ls(y, mu, sigma2, shape, nu))
      u        <- ((4*(nu)^(nu/2)*gamma((3+nu)/2)*(dj + nu)^(-(nu+3)/2)) / (gamma(nu/2)*sqrt(pi)*sqrt(sigma2)*dt.ls(y, mu, sigma2, shape, nu)) )*pt(sqrt((3+nu)/(dj+nu))*A,3+nu)

      S1       <- u
      S2       <- (mutij*u + Mtij*E)
      S3       <- (mutij^2*u + Mtij2 + Mtij*mutij*E)

      E00      <- S1
      E01      <- y*S1
      E02      <- y^2*S1
      E10      <- S2
      E20      <- S3
      E11      <- y*S2
      sigma2s  <- nu/(nu + 2)*sigma2
      sigma2ss <- nu/((nu + 1)*(1 + shape^2))*sigma2

      Aux1     <- pSNI2(Lim1,Lim2, mu,  sigma2s, shape, nu+2, type = "T")
      Aux11    <- pSNI2(Lim1,Lim2, mu, sigma2ss,     0, nu+1, type = "T")
      Aux2     <- pSNI2(Lim1,Lim2, mu,   sigma2, shape,   nu, type = "T")

      mu1      <- mu[cc==1]

      np       <- length(mu1)
      aux1MomW <- aux2MomS <- aux3MomS <- matrix(0, np, 2)

      for(j in 1:np){
        A1a          <- MomenSNI2023(mu1[j],  sigma2s, 0, nu+2, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "ST")
        A2a          <- MomenSNI2023(mu1[j], sigma2ss, 0, nu+1, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "ST")
        A3a          <- MomenSNI2023(mu1[j],   sigma2, 0,   nu, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "ST")
        aux1MomW[j,] <- c(A1a$EUY1, A1a$EUY2)
        aux2MomS[j,] <- c(A2a$EUY1, A2a$EUY2)
        aux3MomS[j,] <- c(A3a$EUY1, A3a$EUY2)
      }

      P1aux <- P2aux <- WW <- u
      P1aux[cc==1]   <- aux1MomW[,1]
      P2aux[cc==1]   <- aux1MomW[,2]
      WW[cc==1]      <- aux2MomS[,1]

      Wphi           <- Aux11/Aux2
      E00Aux         <- Aux1/Aux2
      E01Aux         <- Aux1/Aux2*P1aux
      E02Aux         <- Aux1/Aux2*P2aux
      E10Aux         <- Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + Mtij*cnu*Wphi
      E20Aux         <- (Delta/(Gama + Delta^2))^2*(E02Aux - 2*E01Aux*mu + mu^2*E00Aux) + Mtij*(Delta/(Gama + Delta^2))*cnu*Wphi*(WW - mu) + Mtij2
      E11Aux         <- Delta/(Gama + Delta^2)*(E02Aux - E01Aux*mu) + Mtij*Wphi*cnu*WW

      ychap[cc==1]   <- aux3MomS[,1]
      E00[cc==1]     <- E00Aux[cc==1]
      E01[cc==1]     <- E01Aux[cc==1]
      E02[cc==1]     <- E02Aux[cc==1]
      E10[cc==1]     <- E10Aux[cc==1]
      E20[cc==1]     <- E20Aux[cc==1]
      E11[cc==1]     <- E11Aux[cc==1]

      beta           <- solve(t(x)%*%diag(c(E00))%*%x)%*%(t(x)%*%matrix(E01 - E10*Delta, n, 1))
      mu             <- x%*%beta
      Delta          <- 0
      Gama           <- sum(E02 - 2*E01*mu + E00*mu^2 + Delta^2*E20 - 2*Delta*E11 + 2*Delta*E10*mu)/n
      sigma2         <- Gama + Delta^2
      shape          <- ((sigma2^(-1/2))*Delta)/(sqrt(1 - (Delta^2)*(sigma2^(-1))))

      f <- function(nu){
        sum(log(dTMod(cc, y, mu, sigma2, nu, cens=cens, LS=Lim2)))
      }

      nu             <- optim(par = 0, fn = f, control = list(fnscale = -1), method = "L-BFGS-B", lower = 2.01, upper = 50)$par
      lk             <- f(nu)
      logver         <- lk
      criterio       <- abs((lk/lkante - 1))
      lkante         <- lk

      if (cont==iter.max){
        criterio     <- error/10
      }
    }

    teta_novo        <- matrix(c(beta, sigma2,    nu), ncol = 1)
    teta_novo1       <- matrix(c(beta, sigma2, 0, nu), ncol = 1)

    #########################
    ##  Information Matrix
    #########################

    sbeta            <- c()
    ssigma2          <- c()
    MIE              <- matrix(0, p+1, p+1)
    S                <- matrix(0,   1, p+1)
    sigma            <- sqrt(sigma2)
    lambda           <- 0

    for(i in 1:n){
      sbeta          <- ((1 + lambda^2)/sigma2)*(E01[i]*t(x[i,]) - E00[i]*t(x[i,])*mu[i] - Delta*E10[i]*t(x[i,]))
      ssigma2        <- -1/(2*sigma2) + ((1 + lambda^2)/(2*sigma2^2))*(E02[i] - 2*E01[i]*mu[i] + (t(mu[i])%*%mu[i])*E00[i]) - ((lambda*sqrt(1+lambda^2))/(2*sigma^3))*(E11[i] - E10[i]*mu[i])
      S              <- c(sbeta, ssigma2)
      MIE1           <- S%*%t(S)
      ind            <- lower.tri(MIE1)
      MIE1[ind]      <- t(MIE1)[ind]
      MIE            <- MIE1 + MIE
    }

    se               <- sqrt(diag(solve(MIE)))

    resmt <- NULL
    if(show.envelope){
      envelop        <- EnvelopeRMT_ggp(teta_novo1, y, x, cc, type = "T", family = "ST")
      print(envelop)
      resmt          <- envelop$plot_env$res$resmt
    }
  }

  #########################
  ## Skew-Normal
  #########################

  if(family=="SN"){

    cont       <- 0
    criterio   <- 1
    lkante     <- 1
    ychap      <- y
    n.par      <- p + 2
      y1       <- y[cc==1]
    Lim1aux    <- Lim1[cc==1]
    Lim2aux    <- Lim2[cc==1]

    b          <- -sqrt(2/pi)
    mu         <- muaux + b*Delta

    while(criterio > error){

      cont     <- (cont+1)
      # print(cont)
      # print(c(beta,sigma2,shape))

      Mtij2    <- 1/(1 + (Delta^2)*(Gama^(-1)))
      Mtij     <- sqrt(Mtij2)
      mutij    <- Mtij2*Delta*(Gama^(-1))*(y - mu)
      A        <- mutij / Mtij
      cnu      <- 2/sqrt(2*pi*(1 + shape^2))
      prob     <- pnorm(mutij/Mtij)

      if(length(which(prob == 0)) > 0) prob[which(prob == 0)] <- .Machine$double.xmin

      E        <- dnorm(mutij/Mtij)/prob
      u        <- rep(1, n)

      S1       <- u
      S2       <- ((mutij + b)*u + Mtij*E)
      S3       <- ((mutij + b)^2*u + Mtij2 + Mtij*(mutij+2*b)*E)

      E00      <- S1
      E01      <- y*S1
      E02      <- y^2*S1
      E10      <- S2
      E20      <- S3
      E11      <- y*S2

      sigma2s  <- (1/(1 + shape^2))*sigma2

      Aux11    <- pSNI2(Lim1, Lim2, mu, sigma2s,     0, NULL, type = "SN")
      Aux2     <- pSNI2(Lim1, Lim2, mu,  sigma2, shape, NULL, type = "SN")
      mu1      <- mu[cc==1]

      np       <- length(mu1)
      aux1MomW <- aux2MomS <- aux3MomS <- matrix(0, np, 2)

      if(np > 0){
        for(j in 1:np){
          A1a          <- MomenSNI2023(mu1[j],  sigma2, shape, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A2a          <- MomenSNI2023(mu1[j], sigma2s,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A3a          <- MomenSNI2023(mu1[j],  sigma2,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          aux1MomW[j,] <- c(A1a$EUY1, A1a$EUY2)
          aux2MomS[j,] <- c(A2a$EUY1, A2a$EUY2)
          aux3MomS[j,] <- c(A3a$EUY1, A3a$EUY2)
        }

      Wphi           <- Aux11/Aux2
      E00Aux         <- E01Aux <- E02Aux <- WW1 <- u
      E01Aux[cc==1]  <- aux1MomW[,1]
      E02Aux[cc==1]  <- aux1MomW[,2]
      WW1[cc==1]     <- aux2MomS[,1]
      E10Aux         <- Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + Mtij*cnu*Wphi + b*E00Aux
      E20Aux         <- (Delta/(Gama + Delta^2))^2*(E02Aux - 2*E01Aux*mu + mu^2*E00Aux) + 2*b*(Delta/(Gama + Delta^2))*(E01Aux-E00Aux*mu) + b^2*E00Aux + Mtij2 + Mtij*((Delta/(Gama + Delta^2))*cnu*Wphi*(WW1 - mu)+2*b*cnu*Wphi)
      E11Aux         <- Delta/(Gama + Delta^2)*(E02Aux - E01Aux*mu) + b*E01Aux +Mtij*Wphi*cnu*WW1

      ychap[cc==1]   <- aux3MomS[,1]
      E00[cc==1]     <- E00Aux[cc==1]        # E[U]
      E01[cc==1]     <- E01Aux[cc==1]        # E[UY]
      E02[cc==1]     <- E02Aux[cc==1]        # E[UY2]
      E10[cc==1]     <- E10Aux[cc==1]        # E[UT]
      E20[cc==1]     <- E20Aux[cc==1]        # E[UT2]
      E11[cc==1]     <- E11Aux[cc==1]        # E[UTY]

      }

      beta           <- solve(t(x)%*%diag(E00)%*%x)%*%(t(x)%*%(E01 - E10*Delta))
      muaux          <- x%*%beta
      Delta          <- sum(E11 - E10*muaux)/sum(E20)
      Gama           <- sum(E02 - 2*E01*muaux + E00*muaux^2 + Delta^2*E20 - 2*Delta*E11 + 2*Delta*E10*muaux)/n
      sigma2         <- Gama + Delta^2
      shape          <- ((sigma2^(-1/2))*Delta)/(sqrt(1 - (Delta^2)*(sigma2^(-1))))

      b              <- -sqrt(2/pi)
      mu             <- muaux + b*Delta

      auxpdf         <- dSN(y[cc==0], mu[cc==0], sigma2, shape)
      if(length(which(auxpdf == 0)) > 0) auxpdf[which(auxpdf == 0)] <- .Machine$double.xmin
      if (np > 0) {
        auxcdf <- pSNI2(Lim1[cc == 1], Lim2[cc == 1], mu[cc == 1], sigma2, shape, NULL, type = "SN")
        lk <- sum(log(auxpdf)) + sum(log(auxcdf))
      } else {
        lk <- sum(log(auxpdf))
      }

      logver         <- lk
      criterio       <- abs((lk/lkante - 1))
      lkante         <- lk

      if (cont==iter.max){
        criterio     <- error/10
      }
    }

    ychap            <- E01

    teta_novo        <- matrix(c(beta, sigma2, shape), ncol = 1)

    #########################
    ##  Information Matrix
    #########################

    sbeta            <- c()
    ssigma2          <- c()
    slambda          <- c()
    MIE              <- matrix(0, p+2, p+2)
    S                <- matrix(0,   1, p+2)
    sigma            <- sqrt(sigma2)
    lambda           <- shape

    for(i in 1:n){
      sbeta          <- ((1 + lambda^2)/sigma2)*(E01[i]*t(x[i,]) - E00[i]*t(x[i,])*muaux[i] - Delta*E10[i]*t(x[i,]))
      ssigma2        <- -1/(2*sigma2) + ((1 + lambda^2)/(2*sigma2^2))*(E02[i] - 2*E01[i]*muaux[i] + (t(muaux[i])%*%muaux[i])*E00[i]) - ((lambda*sqrt(1+lambda^2))/(2*sigma^3))*(E11[i] - E10[i]*muaux[i])
      slambda        <- lambda/(1 + lambda^2) - (lambda/sigma2)*(E02[i] - 2*E01[i]*muaux[i] + E00[i]*(t(muaux[i])%*%muaux[i])) + ((1 + 2*lambda^2)/(sigma*sqrt(1+lambda^2)))*(E11[i] - E10[i]*muaux[i]) - lambda*E20[i]
      S              <- c(sbeta, ssigma2, slambda)
      MIE1           <- S%*%t(S)
      ind            <- lower.tri(MIE1)
      MIE1[ind]      <- t(MIE1)[ind]
      MIE            <- MIE1 + MIE
    }

    se               <- sqrt(diag(solve(MIE)))

    resmt <- NULL
    if(show.envelope){
      envelop        <- EnvelopeRMT_ggp(teta_novo, y, x, cc, type = "SN", family = "SN")
      print(envelop)
      resmt          <- envelop$plot_env$res$resmt
    }
  }

  #########################
  ## Normal
  #########################

  if(family=="N"){

    cont       <- 0
    criterio   <- 1
    lkante     <- 1
    shape      <- 0
    ychap      <- y
    n.par      <- p + 1

    y1         <- y[cc==1]
    Lim1aux    <- Lim1[cc==1]
    Lim2aux    <- Lim2[cc==1]

    mu         <- muaux

    while(criterio > error){

      cont     <- (cont+1)
      # print(cont)
      # print(c(beta,sigma2,shape))

      dj       <- ((y - mu)/sqrt(sigma2))^2
      Mtij2    <- 1/(1 + (Delta^2)*(Gama^(-1)))
      Mtij     <- sqrt(Mtij2)
      mutij    <- Mtij2*Delta*(Gama^(-1))*(y - mu)
      A        <- mutij / Mtij
      cnu      <- 2/sqrt(2*pi*(1 + shape^2))
      prob     <- pnorm(mutij/Mtij)

      if(length(which(prob == 0)) > 0) prob[which(prob == 0)] <- .Machine$double.xmin

      E        <- dnorm(mutij/Mtij)/prob
      u        <- rep(1, n)

      S1       <- u
      S2       <- (mutij*u + Mtij*E)
      S3       <- (mutij^2*u + Mtij2 + Mtij*mutij*E)

      E00      <- S1
      E01      <- y*S1
      E02      <- y^2*S1
      E10      <- S2
      E20      <- S3
      E11      <- y*S2

      sigma2s  <- (1/(1 + shape^2))*sigma2

      Aux11    <- pSNI2(Lim1, Lim2, mu, sigma2s,     0, NULL, type = "N")
      Aux2     <- pSNI2(Lim1, Lim2, mu,  sigma2, shape, NULL, type = "N")

      mu1      <- mu[cc==1]

      np       <- length(mu1)
      aux1MomW <- aux2MomS <- matrix(0, np, 2)

      for(j in 1:np){
        if (Lim1aux[j] == -(.Machine$double.xmin)) {
          Lim1aux[j] <- -Inf
        }

        A1a          <- MomenSNI2023(mu1[j],  sigma2, shape, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
        A2a          <- MomenSNI2023(mu1[j], sigma2s,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
        aux1MomW[j,] <- c(A1a$EUY1, A1a$EUY2)
        aux2MomS[j,] <- c(A2a$EUY1, A2a$EUY2)
      }

      Wphi           <- Aux11/Aux2
      E00Aux         <- E01Aux <- E02Aux <- WW1 <- u
      E01Aux[cc==1]  <- aux1MomW[,1]
      E02Aux[cc==1]  <- aux1MomW[,2]
      WW1[cc==1]     <- aux2MomS[,1]
      E10Aux         <- Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + Mtij*cnu*Wphi
      E20Aux         <- (Delta/(Gama + Delta^2))^2*(E02Aux - 2*E01Aux*mu + mu^2*E00Aux) + Mtij*(Delta/(Gama + Delta^2))*cnu*Wphi*(WW1 - mu) + Mtij2
      E11Aux         <- Delta/(Gama + Delta^2)*(E02Aux - E01Aux*mu) + Mtij*Wphi*cnu*WW1

      E00[cc==1]     <- E00Aux[cc==1]
      E01[cc==1]     <- E01Aux[cc==1]
      E02[cc==1]     <- E02Aux[cc==1]
      E10[cc==1]     <- E10Aux[cc==1]
      E20[cc==1]     <- E20Aux[cc==1]
      E11[cc==1]     <- E11Aux[cc==1]

      beta           <- solve(t(x)%*%diag(E00)%*%x)%*%(t(x)%*%(E01 - E10*Delta))
      mu             <- x%*%beta
      Delta          <- 0
      Gama           <- sum(E02 - 2*E01*mu + E00*mu^2 + Delta^2*E20 - 2*Delta*E11 + 2*Delta*E10*mu)/n
      sigma2         <- Gama + Delta^2
      shape          <- ((sigma2^(-1/2))*Delta)/(sqrt(1 - (Delta^2)*(sigma2^(-1))))

      auxpdf         <- dSNMod(   y[cc==0], mu[cc==0], sigma2, shape)
      auxcdf         <- pSNI2(Lim1[cc==1], Lim2[cc==1], mu[cc==1], sigma2, shape, NULL, type = "SN")
      lk             <- sum(log(auxpdf)) + sum(log(auxcdf))
      logver         <- lk
      criterio       <- abs((lk/lkante - 1))
      lkante         <- lk

      if (cont==iter.max){
        criterio     <- error/10
      }
    }

    ychap            <- E01

    teta_novo        <- matrix(c(beta, sigma2   ), ncol = 1)
    teta_novo1       <- matrix(c(beta, sigma2, 0), ncol = 1)

    #########################
    ##  Information Matrix
    #########################

    sbeta            <- c()
    ssigma2          <- c()
    MIE              <- matrix(0, p+1, p+1)
    S                <- matrix(0,   1, p+1)
    sigma            <- sqrt(sigma2)
    lambda           <- 0

    for(i in 1:n){
      sbeta          <- ((1 + lambda^2)/sigma2)*(E01[i]*t(x[i,]) - E00[i]*t(x[i,])*mu[i] - Delta*E10[i]*t(x[i,]))
      ssigma2        <- -1/(2*sigma2) + ((1 + lambda^2)/(2*sigma2^2))*(E02[i] - 2*E01[i]*mu[i] + (t(mu[i])%*%mu[i])*E00[i]) - ((lambda*sqrt(1 + lambda^2))/(2*sigma^3))*(E11[i] - E10[i]*mu[i])
      S              <- c(sbeta, ssigma2)
      MIE1           <- S%*%t(S)
      ind            <- lower.tri(MIE1)
      MIE1[ind]      <- t(MIE1)[ind]
      MIE            <- MIE1 + MIE
    }

    se               <- sqrt(diag(solve(MIE)))

    resmt <- NULL
    if(show.envelope){
      envelop        <- EnvelopeRMT_ggp(teta_novo1, y, x, cc, type = "N", family = "SN")
      print(envelop)
      resmt          <- envelop$plot_env$res$resmt
    }
  }

  #########################
  ## Skew-Contaminated Normal
  #########################

  if(family=="SCN"){

    cont       <- 0
    criterio   <- 1
    lkante     <- 1
    ychap      <- y
    n.par      <- p + 2

    y1         <- y[cc==1]
    Lim1aux    <- Lim1[cc==1]
    Lim2aux    <- Lim2[cc==1]

    k1         <- (nu[1]/(nu[2]^(0.5))) + 1-nu[1]
    b          <- -sqrt(2/pi)*k1
    mu         <- muaux + b*Delta

    while(criterio > error){

      cont     <- (cont + 1)
      # print(cont)
      # print(c(beta,sigma2,shape,nu))

      Mtij2    <- 1/(1 + (Delta^2)*(Gama^(-1)))
      Mtij     <- sqrt(Mtij2)
      mutij    <- Mtij2*Delta*(Gama^(-1))*(y - mu)
      A        <- mutij / Mtij
      cnu1     <- (nu[1]*(nu[2])^(0.5))/sqrt(2*pi*(1+shape^2))
      cnu2     <- (1-nu[1])/sqrt(2*pi*(1+shape^2))

      u        <- 2/(DenSCN(y, mu, sigma2, shape, nu))*(nu[1]*nu[2]*dnorm(y, mu, sqrt(sigma2/nu[2]))*pnorm(sqrt(nu[2])*shape*sigma2^(-0.5)*(y-mu)) + (1 - nu[1])*dnorm(y, mu, sqrt(sigma2))*pnorm(shape*sigma2^(-0.5)*(y-mu)))
      E        <- 2/(DenSCN(y, mu, sigma2, shape, nu))*(nu[1]*(nu[2])^(0.5)*dnorm(y, mu, sqrt(sigma2/nu[2]))*dnorm(sqrt(nu[2])*shape*sigma2^(-0.5)*(y-mu)) + (1 - nu[1])*dnorm(y, mu, sqrt(sigma2))*dnorm(shape*sigma2^(-0.5)*(y-mu)))

      if (any(is.na(u[cc == 0]))){
        at    <- which(is.na(u) & cc==0)
        u[at] <- 2/(.Machine$double.xmin)*(nu[1]*nu[2]*.Machine$double.xmin*pnorm(sqrt(nu[2])*shape*sigma2^(-0.5)*(y[at]-mu[at])) + (1 - nu[1])*.Machine$double.xmin*pnorm(shape*sigma2^(-0.5)*(y[at]-mu[at])))
      }
      if (any(is.na(E[cc == 0]))){
        at    <- which(is.na(E) & cc==0)
        E[at] <- 2/(.Machine$double.xmin)*(nu[1]*(nu[2])^(0.5)*.Machine$double.xmin*.Machine$double.xmin + (1 - nu[1])*.Machine$double.xmin*.Machine$double.xmin)
      }

      S1       <- u
      S2       <- ((mutij + b)*u + Mtij*E)
      S3       <- ((mutij + b)^2*u + Mtij2 + Mtij*(mutij+2*b)*E)

      E00      <- S1
      E01      <- y*S1
      E02      <- y^2*S1
      E10      <- S2
      E20      <- S3
      E11      <- y*S2

      AuxDen    <- 1/(1+shape^2)
      sigma2s   <- sigma2/nu[2]
      sigma2ss  <- (sigma2/nu[2])*AuxDen
      sigma2sss <- sigma2*AuxDen

      Aux1     <- pSNI2(Lim1,    Lim2, mu,   sigma2s,   shape, NULL, type = "SN")
      Aux11    <- pSNI2(Lim1,    Lim2, mu,    sigma2,   shape, NULL, type = "SN")
      Aux2     <- pSNI_NC(Lim1, Lim2, mu,    sigma2,   shape,   nu, type = "SCN")
      Aux3     <- pSNI2(Lim1,    Lim2, mu,  sigma2ss,       0, NULL, type = "SN")
      Aux31    <- pSNI2(Lim1,    Lim2, mu, sigma2sss,       0, NULL, type = "SN")


      mu1       <- mu[cc==1]
      np        <- length(mu1)
      aux1MomW1 <- aux2MomW2 <- aux3MomW3 <- aux4MomW4 <- matrix(0, np, 2)

      if(np > 0){
        for(j in 1:np){
          A1a           <- MomenSNI2023(mu1[j],    sigma2, shape, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A2a           <- MomenSNI2023(mu1[j],   sigma2s, shape, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A3a           <- MomenSNI2023(mu1[j],  sigma2ss,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A4a           <- MomenSNI2023(mu1[j], sigma2sss,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")

          aux1MomW1[j,] <- c(A1a$EUY1, A1a$EUY2)
          aux2MomW2[j,] <- c(A2a$EUY1, A2a$EUY2)
          aux3MomW3[j,] <- c(A3a$EUY1, A3a$EUY2)
          aux4MomW4[j,] <- c(A4a$EUY1, A4a$EUY2)
        }
      }

      P1aux <- P2aux  <- P3aux <- P4aux <- P5aux <- P6aux <- WW <- u
      P1aux[cc==1]    <- aux1MomW1[,1]
      P2aux[cc==1]    <- aux1MomW1[,2]
      P3aux[cc==1]    <- aux2MomW2[,1]
      P4aux[cc==1]    <- aux2MomW2[,2]
      P5aux[cc==1]    <- aux3MomW3[,1]
      P6aux[cc==1]    <- aux4MomW4[,1]

      AuxT0          <- (2/Aux2)*(cnu1*Aux3 + cnu2*Aux31)
      AuxT1          <- (2/Aux2)*(cnu1*Aux3*P5aux + cnu2*Aux31*P6aux)

      # Com censura
      E00Aux         <- (1/Aux2)*(nu[1]*nu[2]*Aux1+(1-nu[1])*Aux11)
      E01Aux         <- (1/Aux2)*(nu[1]*nu[2]*Aux1*P3aux+(1-nu[1])*Aux11*P1aux)
      E02Aux         <- (1/Aux2)*(nu[1]*nu[2]*Aux1*P4aux+(1-nu[1])*Aux11*P2aux)
      E10Aux         <- Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + b*E00Aux + Mtij*AuxT0
      E20Aux         <- (Delta/(Gama + Delta^2))^2*(E02Aux - 2*E01Aux*mu + mu^2*E00Aux) + 2*b*(Delta/(Gama + Delta^2))*(E01Aux - E00Aux*mu) + b^2*E00Aux + Mtij2 + Mtij*((Delta/(Gama + Delta^2))*(AuxT1-mu*AuxT0) + 2*b*AuxT0 )
      E11Aux         <- Delta/(Gama + Delta^2)*(E02Aux - E01Aux*mu) + b*E01Aux + Mtij*AuxT1

      E00[cc==1]     <- E00Aux[cc==1]
      E01[cc==1]     <- E01Aux[cc==1]
      E02[cc==1]     <- E02Aux[cc==1]
      E10[cc==1]     <- E10Aux[cc==1]
      E20[cc==1]     <- E20Aux[cc==1]
      E11[cc==1]     <- E11Aux[cc==1]

      beta           <- solve(t(x)%*%diag(c(E00))%*%x)%*%(t(x)%*%matrix(E01 - E10*Delta, n, 1))
      muaux          <- x%*%beta
      Delta          <- sum(E11 - E10*muaux)/sum(E20)
      Gama           <- sum(E02 - 2*E01*muaux + E00*muaux^2 + Delta^2*E20 - 2*Delta*E11 + 2*Delta*E10*muaux)/n
      sigma2         <- Gama + Delta^2
      shape          <- ((sigma2^(-1/2))*Delta)/(sqrt(1 - (Delta^2)*(sigma2^(-1))))

      k1             <- (nu[1]/(nu[2]^(0.5))) + 1-nu[1]
      b              <- -sqrt(2/pi)*k1
      mu             <- muaux + b*Delta

      f<-function(nu){
        ver <- sum(log(dSCNMod(cc, y, mu, sigma2 , shape, nu, cens=cens, LS=Lim2)))
        return(-ver)
      }

      if (nu[2] == 0) {
        nu[2] <- .Machine$double.xmin
      }

      nu <- optim(c(0.5, 0.5), f, method = "L-BFGS-B", lower = c(0.01, 0.01), upper = c(0.99, 0.99))$par

      nu <- nu

      lk             <- -f(nu)
      logver         <- lk
      criterio       <- abs((lk/lkante - 1))
      lkante         <- lk

      if (cont==iter.max){
        criterio     <- error/10
      }
    }

    teta_novo        <- matrix(c(beta, sigma2, shape, nu), ncol = 1) # to compute the number of parameters


    #########################
    ##  Information Matrix
    #########################

    sbeta            <- c()
    ssigma2          <- c()
    sshape           <- c()
    MIE              <- matrix(0, p+2, p+2)
    S                <- matrix(0,   1, p+2)
    sigma            <- sqrt(sigma2)
    lambda           <- shape

    for(i in 1:n){
      sbeta          <- ((1 + lambda^2)/sigma2)*(E01[i]*t(x[i,]) - E00[i]*t(x[i,])*muaux[i] - Delta*E10[i]*t(x[i,]))
      ssigma2        <- -1/(2*sigma2) + ((1 + lambda^2)/(2*sigma2^2))*(E02[i] - 2*E01[i]*muaux[i] + (t(muaux[i])%*%muaux[i])*E00[i]) - ((lambda*sqrt(1+shape^2))/(2*sigma^3))*(E11[i] - E10[i]*muaux[i])
      sshape         <- lambda/(1 + lambda^2) - (lambda/sigma2)*(E02[i] - 2*E01[i]*muaux[i] + E00[i]*(t(muaux[i])%*%muaux[i])) + ((1 + 2*lambda^2)/(sigma*sqrt(1+lambda^2)))*(E11[i] - E10[i]*muaux[i]) - lambda*E20[i]
      S              <- c(sbeta, ssigma2, sshape)
      MIE1           <- S%*%t(S)
      ind            <- lower.tri(MIE1)
      MIE1[ind]      <- t(MIE1)[ind]
      MIE            <- MIE1 + MIE
    }

    se <- tryCatch(
      {
        sqrt(diag(solve(MIE)))
      },
      error = function(e) {
        sqrt(diag(chol2inv(chol(MIE))))
      }
    )

    resmt <- NULL
    if(show.envelope){
      envelop        <- EnvelopeRMT_ggp(teta_novo, y, x, cc, type = "SCN", family = "SCN")
      print(envelop)
      resmt          <- envelop$plot_env$res$resmt
    }
  }

  if(family=="CN"){

    cont       <- 0
    criterio   <- 1
    lkante     <- 1
    ychap      <- y
    n.par      <- p + 1

    y1         <- y[cc==1]
    Lim1aux    <- Lim1[cc==1]
    Lim2aux    <- Lim2[cc==1]

    k1         <- (nu[1]/(nu[2]^(0.5))) + 1-nu[1]
    b          <- -sqrt(2/pi)*k1
    mu         <- muaux + b*Delta

    while(criterio > error){

      cont     <- (cont + 1)
      print(cont)
      print(c(beta,sigma2,shape,nu))

      Mtij2    <- 1/(1 + (Delta^2)*(Gama^(-1)))
      Mtij     <- sqrt(Mtij2)
      mutij    <- Mtij2*Delta*(Gama^(-1))*(y - mu)
      A        <- mutij / Mtij
      cnu1     <- (nu[1]*(nu[2])^(0.5))/sqrt(2*pi*(1+shape^2))
      cnu2     <- (1-nu[1])/sqrt(2*pi*(1+shape^2))

      u        <- 2/(DenSCN(y, mu, sigma2, shape, nu))*(nu[1]*nu[2]*dnorm(y, mu, sqrt(sigma2/nu[2]))*pnorm(sqrt(nu[2])*shape*sigma2^(-0.5)*(y-mu)) + (1 - nu[1])*dnorm(y, mu, sqrt(sigma2))*pnorm(shape*sigma2^(-0.5)*(y-mu)))
      E        <- 2/(DenSCN(y, mu, sigma2, shape, nu))*(nu[1]*(nu[2])^(0.5)*dnorm(y, mu, sqrt(sigma2/nu[2]))*dnorm(sqrt(nu[2])*shape*sigma2^(-0.5)*(y-mu)) + (1 - nu[1])*dnorm(y, mu, sqrt(sigma2))*dnorm(shape*sigma2^(-0.5)*(y-mu)))

      if (any(is.na(u[cc == 0]))){
        at     <- which(is.na(u) & cc==0)
        u[at]  <- 2/(.Machine$double.xmin)*(nu[1]*nu[2]*.Machine$double.xmin*pnorm(sqrt(nu[2])*shape*sigma2^(-0.5)*(y[at]-mu[at])) + (1 - nu[1])*.Machine$double.xmin*pnorm(shape*sigma2^(-0.5)*(y[at]-mu[at])))
      }
      if (any(is.na(E[cc == 0]))){
        at     <- which(is.na(E) & cc==0)
        E[at]  <- 2/(.Machine$double.xmin)*(nu[1]*(nu[2])^(0.5)*.Machine$double.xmin*.Machine$double.xmin + (1 - nu[1])*.Machine$double.xmin*.Machine$double.xmin)
      }

      S1       <- u
      S2       <- ((mutij + b)*u + Mtij*E)
      S3       <- ((mutij + b)^2*u + Mtij2 + Mtij*(mutij+2*b)*E)

      E00      <- S1
      E01      <- y*S1
      E02      <- y^2*S1
      E10      <- S2
      E20      <- S3
      E11      <- y*S2

      AuxDen    <- 1/(1+shape^2)
      sigma2s   <- sigma2/nu[2]
      sigma2ss  <- (sigma2/nu[2])*AuxDen
      sigma2sss <- sigma2*AuxDen

      Aux1     <- pSNI2(Lim1,    Lim2, mu,   sigma2s,   shape, NULL, type = "SN")
      Aux11    <- pSNI2(Lim1,    Lim2, mu,    sigma2,   shape, NULL, type = "SN")
      Aux2     <- pSNI_NC(Lim1, Lim2, mu,    sigma2,   shape,   nu, type = "SCN")
      Aux3     <- pSNI2(Lim1,    Lim2, mu,  sigma2ss,       0, NULL, type = "SN")
      Aux31    <- pSNI2(Lim1,    Lim2, mu, sigma2sss,       0, NULL, type = "SN")

      mu1       <- mu[cc==1]
      np        <- length(mu1)
      aux1MomW1 <- aux2MomW2 <- aux3MomW3 <- aux4MomW4 <- matrix(0, np, 2)

      if(np > 0){
        for(j in 1:np){
          A1a           <- MomenSNI2023(mu1[j],    sigma2, shape, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A2a           <- MomenSNI2023(mu1[j],   sigma2s, shape, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A3a           <- MomenSNI2023(mu1[j],  sigma2ss,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")
          A4a           <- MomenSNI2023(mu1[j], sigma2sss,     0, NULL, delta=NULL, Lim1=Lim1aux[j], Lim2=Lim2aux[j], type = "SN")

          aux1MomW1[j,] <- c(A1a$EUY1, A1a$EUY2)
          aux2MomW2[j,] <- c(A2a$EUY1, A2a$EUY2)
          aux3MomW3[j,] <- c(A3a$EUY1, A3a$EUY2)
          aux4MomW4[j,] <- c(A4a$EUY1, A4a$EUY2)
        }
      }

      P1aux <- P2aux  <- P3aux <- P4aux <- P5aux <- P6aux <- WW <- u
      P1aux[cc==1]    <- aux1MomW1[,1]
      P2aux[cc==1]    <- aux1MomW1[,2]
      P3aux[cc==1]    <- aux2MomW2[,1]
      P4aux[cc==1]    <- aux2MomW2[,2]
      P5aux[cc==1]    <- aux3MomW3[,1]
      P6aux[cc==1]    <- aux4MomW4[,1]

      AuxT0          <- (2/Aux2)*(cnu1*Aux3 + cnu2*Aux31)
      AuxT1          <- (2/Aux2)*(cnu1*Aux3*P5aux + cnu2*Aux31*P6aux)

      E00Aux         <- (1/Aux2)*(nu[1]*nu[2]*Aux1+(1-nu[1])*Aux11)
      E01Aux         <- (1/Aux2)*(nu[1]*nu[2]*Aux1*P3aux+(1-nu[1])*Aux11*P1aux)
      E02Aux         <- (1/Aux2)*(nu[1]*nu[2]*Aux1*P4aux+(1-nu[1])*Aux11*P2aux)
      E10Aux         <- Delta/(Gama + Delta^2)*(E01Aux - E00Aux*mu) + b*E00Aux + Mtij*AuxT0
      E20Aux         <- (Delta/(Gama + Delta^2))^2*(E02Aux - 2*E01Aux*mu + mu^2*E00Aux) + 2*b*(Delta/(Gama + Delta^2))*(E01Aux - E00Aux*mu) + b^2*E00Aux + Mtij2 + Mtij*((Delta/(Gama + Delta^2))*(AuxT1-mu*AuxT0) + 2*b*AuxT0 )
      E11Aux         <- Delta/(Gama + Delta^2)*(E02Aux - E01Aux*mu) + b*E01Aux + Mtij*AuxT1

      E00[cc==1]     <- E00Aux[cc==1]
      E01[cc==1]     <- E01Aux[cc==1]
      E02[cc==1]     <- E02Aux[cc==1]
      E10[cc==1]     <- E10Aux[cc==1]
      E20[cc==1]     <- E20Aux[cc==1]
      E11[cc==1]     <- E11Aux[cc==1]

      beta           <- solve(t(x)%*%diag(c(E00))%*%x)%*%(t(x)%*%matrix(E01 - E10*Delta, n, 1))
      muaux          <- x%*%beta
      Delta          <- 0
      Gama           <- sum(E02 - 2*E01*muaux + E00*muaux^2 + Delta^2*E20 - 2*Delta*E11 + 2*Delta*E10*muaux)/n
      sigma2         <- Gama + Delta^2
      shape          <- ((sigma2^(-1/2))*Delta)/(sqrt(1 - (Delta^2)*(sigma2^(-1))))

      k1             <- (nu[1]/(nu[2]^(0.5))) + 1-nu[1]
      b              <- -sqrt(2/pi)*k1
      mu             <- muaux + b*Delta

      f<-function(nu){
        ver <- sum(log(dSCNMod(cc, y, mu, sigma2 , shape, nu, cens=cens, LS=Lim2)))
        return(-ver)
      }

      if (nu[2] == 0) {
        nu[2] <- .Machine$double.xmin
      }

      nu <- optim(c(0.5, 0.5), f, method = "L-BFGS-B", lower = c(0.01, 0.01), upper = c(0.99, 0.99))$par

      nu <- nu

      lk             <- -f(nu)
      logver         <- lk
      criterio       <- abs((lk/lkante - 1))
      lkante         <- lk

      if (cont==iter.max){
        criterio     <- error/10
      }
    }

    teta_novo        <- matrix(c(beta, sigma2, nu), ncol = 1) # to compute the number of parameters
    teta_novo1       <- matrix(c(beta, sigma2, 0, nu), ncol = 1)

    #########################
    ##  Information Matrix
    #########################

    sbeta            <- c()
    ssigma2          <- c()
    MIE              <- matrix(0, p+1, p+1)
    S                <- matrix(0,   1, p+1)
    sigma            <- sqrt(sigma2)
    lambda           <- 0

    for(i in 1:n){
      sbeta          <- ((1 + lambda^2)/sigma2)*(E01[i]*t(x[i,]) - E00[i]*t(x[i,])*muaux[i] - Delta*E10[i]*t(x[i,]))
      ssigma2        <- -1/(2*sigma2) + ((1 + lambda^2)/(2*sigma2^2))*(E02[i] - 2*E01[i]*muaux[i] + (t(muaux[i])%*%muaux[i])*E00[i]) - ((lambda*sqrt(1+shape^2))/(2*sigma^3))*(E11[i] - E10[i]*muaux[i])
      S              <- c(sbeta, ssigma2)
      MIE1           <- S%*%t(S)
      ind            <- lower.tri(MIE1)
      MIE1[ind]      <- t(MIE1)[ind]
      MIE            <- MIE1 + MIE
    }

    se <- tryCatch(
      {
        sqrt(diag(solve(MIE)))
      },
      error = function(e) {
        sqrt(diag(chol2inv(chol(MIE))))
      }
    )

    resmt <- NULL
    if(show.envelope){
      envelop        <- EnvelopeRMT_ggp(teta_novo1, y, x, cc, type = "CN", family = "SCN")
      print(envelop)
      resmt          <- envelop$plot_env$res$resmt
    }
  }



  #################################################################

  aic                <- -2*lk +             2*length(teta_novo)
  bic                <- -2*lk +        log(n)*length(teta_novo)
  caic               <- -2*lk +  (log(n) + 1)*length(teta_novo)
  hq                 <- -2*lk + 2*log(log(n))*length(teta_novo)

  #####################################################
  ### Summary of the model results.
  #####################################################

  if(all(x[,1] == 1)) {
    namesx <- ('(Intercept)     ')
    if(ncol(x) > 1){
      for(i in 2:ncol(x)){
        namesx <- cbind(namesx, paste("x", i-1, sep = ""))
      }
    }
  } else {
    # Caso: no tiene intercepto
    namesx <- paste("x", 1:ncol(x), sep = "")
  }


  # namesx <- ('(Intercept)     ')

  # if(ncol(as.matrix(x)) > 1){
  #   for(i in 2:ncol(as.matrix(x))){
  #     namesx <- cbind(namesx, paste("x", i-1,"     ", sep = ""))
  #   }
  # }

  t.est            <- teta_novo[1:n.par]/se[1:n.par]
  p.est            <- 2*(1 - pt(abs(t.est), n-1))

  param.num <- cbind(
    Estimativa = teta_novo[1:n.par],
    SE         = se[1:n.par],
    t          = t.est,
    p.value    = p.est
  )

  # Ahora creamos versión para imprimir
  p.format <- ifelse(p.est < 0.0001,
                     "<0.0001",
                     formatC(p.est, format = "f", digits = 4))

  param <- data.frame(
    Estimate   = round(teta_novo[1:n.par], 3),
    Std.Error  = round(se[1:n.par], 3),
    t.value    = round(t.est, 3),
    `Pr(>|t|)` = p.format,
    row.names  = NULL
  )

  # p.format <- ifelse(p.est < 0.0001,
  #                    formatC(p.est, format = "e", digits = 3),   # científica
  #                    formatC(p.est, format = "f", digits = 4))   # decimal

  # param            <- round(cbind(teta_novo[1:n.par], se[1:n.par], t.est, p.est), digits = 3)
  # param <- cbind(
  #   Estimativa = formatC(teta_novo[1:n.par], format = "f", digits = 3),
  #   SE         = formatC(se[1:n.par], format = "f", digits = 3),
  #   t          = formatC(t.est, format = "f", digits = 3),
  #   p.value    = p.format
  # )
  namespar         <- colnames(x)
  colx             <- ncol(as.matrix(x))

  if(length(namespar) == 0){
    namespar       <- namesx[1:colx]
  }

  if(family=="N" || family=="T" || family=="CN"){
    dimnames(param)  <- list(c(namespar, expression(sigma^2)), c("Estimate", "Std. Error", "t value", "Pr(>|t|)"))
  }

  if(family=="SN" || family=="ST" || family=="SCN"){
    dimnames(param)  <- list(c(namespar, expression(sigma^2), expression(lambda)), c("Estimate", "Std. Error", "t value", "Pr(>|t|)"))
  }

  if(verbose){
  cat('\n')
  cat('-------------------------------------------------------\n')
  cat('                  EM estimates and SE                  \n')
  cat('                                                       \n')
  cat('Coefficients:                                          \n')
  print(param)
  cat('-------------------------------------------------------\n')

  if(family=="T" || family=="ST" ){
    cat("Degree of freedom: ", round(nu, 3), "\n")
  }

  if (family == "CN" || family == "SCN") {
    cat("nu1 ", round(nu[1], 3), " and nu2", round(nu[2], 3), "\n")
  }

  cat('-------------------------------------------------------\n')
  cat('\r \n')
  critFin           <- c(logver, aic, bic, caic, hq)
  critFin           <- round(t(as.matrix(critFin)), digits = 3)
  dimnames(critFin) <- list(c("Value"), c("Loglik", "AIC", "BIC","CAIC", "HQ"))
  cat('\n')
  cat('Model selection criteria                               \n')
  cat('-------------------------------------------------------\n')
  print(critFin)
  cat('-------------------------------------------------------\n')
  cat('\r \n')
  }

  return(list(beta = teta_novo[1:p], sigma2 = teta_novo[p+1], lambda = lambda,
              nu = nu, SE = se,
              iter = cont, logver = logver,
              AIC = aic, BIC = bic, CAIC = caic, HQ = hq,
              residual = resmt))
}
