#' matchSigs
#'
#' This function is used to check to find a mapping between two similar sets of 
#' signatures. It compares the signature values to see how similar the
#' proposed signatures are and shows you the best matches. It uses the 
#' measure of cosine similarity to compare signatures. The two signature sets
#' must have the same underlying components to be matched.
#'
#' @param referenceSigs Signature matrix from your reference analysis
#' @param toCompareSigs Signature matrix from run that you want to compare to reference
#' @return Prints out the signature mapping and returns the avg similarity
#' @export
#' @examples
#' matchSigs(referenceExp$sigs, referenceExp$sigs)
matchSigs = function(referenceSigs, toCompareSigs){
   
   swapped = F
   
   if (nrow(referenceSigs) != nrow(toCompareSigs)){
      stop("Underlying components for the signatures need to be the same. The input
           signature matrices have different number of components.")
   }
   
   if (ncol(referenceSigs) > ncol(toCompareSigs)){
      swapped = T
      sigMat = toCompareSigs
      sigMatVal = referenceSigs
   } else {
      sigMat = referenceSigs
      sigMatVal = toCompareSigs
   }
   

   nSig = ncol(sigMat)
   nvalSig = ncol(sigMatVal)

   cosSim = function(x,y){
      if (sum(x) == 0 || sum(y) == 0)
         return(0)
      else
         return(sum(x*y)/(sqrt(sum(x^2))*sqrt(sum(y^2))))
   }
   
   cors = data.frame(TestSig=1:nSig,ValSig=rep(0,nSig),CorVal=rep(0,nSig))

   #Goes through each of the signatures
   cantUse = numeric()
   i = 1
   while (i <= nSig){
      if (cors[i,2]!=0){  #If the best match has already been found
         i = i+1
         next
      }
      high = -1
      bestMatch = 0
      #Compare it to all other signatures
      for (j in 1:nvalSig){
         if (j %in% cantUse)
            next
            value = cosSim(sigMat[,i],sigMatVal[,j])
         if (abs(value) > high){
            high = value
            bestMatch = j
         }
      }
      #If the best match has already been used
      if (bestMatch %in% cors[,2]){
         #Check if the new corellation is higher and if so overwrite
         if (cors[which(cors[,2]==bestMatch),3] > high){
            cantUse = c(cantUse,bestMatch)
            next
         }
         else {
            prev = which(cors[,2]==bestMatch)
            cors[i,2] = bestMatch
            cors[i,3] = high
            i = prev
            cors[i,2] = 0
            cors[i,3] = 0
         }
      }
      #Otherwise, save the best match and move to next i
      else {
         cantUse = numeric()
         cors[i,2] = bestMatch
         cors[i,3] = high
         i = i+1
      }
   }
   
   if (swapped){
      temp = cors$ValSig
      cors$ValSig = cors$TestSig
      cors$TestSig = temp
   }
   
   message("Average signature similarity: ",mean(cors[,3]),"\n")
   return(cors)
}
