#' Full conditional function for sigma2
#'
#' @param k A scalar for the number of changes points in a cluster
#' @param Yn A vector or matrix with data sequences for a cluster
#' @param Tln A vector with the change-point positions for a cluster
#' @param alphan A vector with the constant level values for each interval between change points for a cluster
#' @param as The hyperparameter value for the shape parameter in the inverse-gamma prior for the variance
#'    component
#' @param bs The hyperparameter value for the scale parameter in the inverse-gamma prior for the variance
#'    component
#' @param M A scalar representing the number of points available for each data sequence
#'
#' @returns A numerical value corresponding to a sampled value from the full conditional of the variance component
#'
#' @note
#' This function is called within the Gibbs sampler, but it can be used separately as well.
#'
#' @seealso [gibbs_alg()]
#'
#' @examples
#' data(data)
#' possigma2n(as = 2, bs = 1000, M = 50, Yn = data[,1], k = 0, Tln = 50, alphan = 15)
#'
#' @export
#'
possigma2n <- function(as, bs, M, Yn, k, Tln, alphan){


  if(k == 0){
    Tln <- M
    Xn <- as.matrix(rep(1, Tln))
  } else{
    Xn <- apply(diag(k+1),2,function(x){rep(x,diff(Tln))})


  }

  an <- (M/2) + as
  bn <- (((t(Yn - Xn%*%alphan)%*%(Yn - Xn%*%alphan))/2) + (1/bs))#rate parameter : exp((1/B)*x) bn = 1/B

  return(extraDistr::rinvgamma(1, alpha = an, beta = bn))
}

