#' @title  Initialize hyperparmeters for BCFM model
#' @description  The function returns a list of hyperparameters of Omega, sigma^2 and mu. It also calls the results from cluster.hyperparms and information from model.attributes.
#'
#' @param model.attributes  Model attributes generated by initialize.model.attributes
#' @param cluster.hyperparms  Cluster related hyperparameters generated by initialize.cluster.hyperparms
#' @param n.sigma  The shape parameter of sigma^2. If not specified, 6.
#' @param n.s2.sigma  The rate parameter of sigma^2. If not specified, 4.
#' @param n.tau  The shape parameter of tau^2. If not specified, 6.
#' @param n.s2.tau  The rate parameter of tau^2. If not specified, 4.
#' @param omega.diag.nu  The shape parameter for the first cluster covariance 
#' @param p.exponent  The Dirichlet priors of probabilities.
#'
#' @return A list of fixed hyperparmeters of mu, Omega and sigma squared.
#' @export

initialize.hyp.parm = function(model.attributes,cluster.hyperparms, n.sigma = 2.2, n.s2.sigma = 0.1, n.tau = 1, n.s2.tau = 1, omega.diag.nu = 2, p.exponent = NA){
  
  hyp.parm = NULL
  
  hyp.parm$omega.nu <- model.attributes$L + 2 # This choice implies that the expected value of Omega is equal to Psi.
  hyp.parm$omega.diag.nu <- omega.diag.nu
  hyp.parm$n.sigma <- n.sigma
  hyp.parm$n.s2.sigma <- n.s2.sigma
  hyp.parm$n.tau <- n.tau
  hyp.parm$n.s2.tau <- n.s2.tau
  hyp.parm$omega.psi = cluster.hyperparms$omega.psi
  hyp.parm$mu.m = cluster.hyperparms$mu.m
  hyp.parm$mu.C = cluster.hyperparms$mu.C
  hyp.parm$initial.cluster = cluster.hyperparms$initial.cluster
  if(is.na(p.exponent[1])){
    hyp.parm$p.exponent <- cluster.hyperparms$p.exponent
  }
  if(!is.na(p.exponent[1]) & length(p.exponent) == 1){
    hyp.parm$p.exponent <- rep(p.exponent, model.attributes$G)
  }
  if(!is.na(p.exponent[1]) & length(p.exponent) == model.attributes$G){
    hyp.parm$p.exponent <- p.exponent
  }
  
  return(hyp.parm)
}
