#' @title  Initialize cluster hyperparameters
#' @description  The function returns list of hyperparmeters for Omegas and mus. 
#'
#' @param data  The dataset.
#' @param model.attributes  Model attributes generated by initialize.model.attributes
#' @param covariance  Use of covariance matrix of common factors. If FALSE, it uses the correlation matrix.
#' @param diag.Psi  Diagonal matrix for cluster covariance. If FALSE, it uses the sample covariance.
#' @param vague.mu  Use of large cluster covariance prior.
#' @param zero.mu  Set the cluster mean prior at 0. If FALSE, the cluster mean prior are the sample means of the clusters.
#' @param seed  Optional integer seed for reproducibility.
#'
#' @return  A list of mean and variance hyperparameter of mu, and scale hyperparameter of Omega
#' @importFrom fastmatrix ldl
#' @export


initialize.cluster.hyperparms = function(data, model.attributes, covariance=FALSE, diag.Psi = FALSE, vague.mu = FALSE, zero.mu = FALSE, seed = NULL){

  threshold <- round(model.attributes$R + 0.5 - 0.5 * sqrt( 1 + 8 * model.attributes$R))
  if(model.attributes$L > threshold){
    stop(paste0(model.attributes$L, " factors are too many for ", model.attributes$R, " variables"))
  }

  # reformat data to long (2 dimensions) form instead of 3 dimension,i.e 200,20,5 to 1000,20
  # wide form 
  data.matrix = c()
  for(tt in 1:model.attributes$times){
    data.matrix = rbind(data.matrix,as.matrix(data[,,tt]))
  }
  
  # Run factor analysis of data.matrix 
  if(covariance){ # covariance
    fa_result <- fa(data.matrix, nfactors = model.attributes$L, fm="minres", rotate = "none", covar=TRUE, warnings=FALSE)
  } else{ # correlation
    fa_result <- fa(data.matrix, nfactors = model.attributes$L, fm="minres", rotate = "none", covar=FALSE, warnings=FALSE)
  }
  eigstacked <- matrix(fa_result$loadings, ncol=model.attributes$L)

  # create trasformation matrix
  if(model.attributes$L > 1){transM <- leading.one(eigstacked)}
  if(model.attributes$L == 1){transM <- leading.one(cbind(eigstacked))}
  
  # apply structural constraint on the factor loading matrix
  B.transformed <- eigstacked %*% transM

  # Estimate common factors
  if(covariance){ # covariance
    uniqueness <- diag(var(data.matrix)) - fa_result$communality
    aux.matrix = solve(t(B.transformed) %*% diag(1/uniqueness) %*% B.transformed) %*% t(B.transformed) %*% diag(1/uniqueness) 
    data.PCX.transformed = data.matrix %*% t(aux.matrix)
  } else{ # correlation
    uniqueness <- 1 - fa_result$communality
    aux.matrix = solve(t(B.transformed) %*% diag(1/uniqueness) %*% B.transformed) %*% t(B.transformed) %*% diag(1/uniqueness) 
    data.PCX.transformed = scale(data.matrix) %*% t(aux.matrix)
  }
  
  # kmeans clustering of the estimated common factors
  
  n.iter <- 50
  kclust.assigned <- matrix(NA, nrow(data.matrix), n.iter)
  kclust.sse <- matrix(NA, nrow(data.matrix), n.iter)
  
  for(i in 1:n.iter){
    if (!is.null(seed)) set.seed(seed)

    currentkmeans <- kmeans(data.PCX.transformed, centers = model.attributes$G) # centers = number of clusters
    for(j in 1:nrow(data.matrix)){
      kclust.sse[j,i] <- sum((data.PCX.transformed[j,] - currentkmeans$centers[currentkmeans$cluster[j],])^2)
    }
    kclust.assigned[,i] <- currentkmeans$cluster
  }
  kclust.dist <- apply(kclust.sse, 2, sum)
  # which.min: Determines the location, i.e., index of the (first) minimum or maximum of a numeric (or logical) vector.
  # finds the cluster with min dist and chooses that col of cluster
  data.clust <- kclust.assigned[,which.min(kclust.dist)]
  # order the frequency of cluster:1,2,3,4 occurence is decreasing order
  data.clust.order = order(table(data.clust), decreasing=TRUE)
  data.clust.new = rep(NA,nrow(data.matrix))
  
  # c(1,2,3,4)[FALSE  TRUE FALSE FALSE]
  for(i in 1:nrow(data.matrix)){
    data.clust.new[i] = (1:model.attributes$G)[data.clust[i] == data.clust.order]
  }
  data.clust = data.clust.new
  p.exponent = rep(2,model.attributes$G)
  
  # prior parameter estimates for the Inverse Wishart Distribution
  Omega.Psi <- array(NA,c(model.attributes$G,model.attributes$L,model.attributes$L)) 
  mu.m <- array(NA,c(model.attributes$G,model.attributes$L))
  mu.C <- array(NA,c(model.attributes$G,model.attributes$L,model.attributes$L))
  
  if(diag.Psi){ # diag does not apply to the first cluster
    # dim data.PCX[data.clust==1,] is 460X3.
    Omega.Psi[1,,] <- cov(data.PCX.transformed[data.clust == 1,])
    # ldl decomposition of a real symmetric matrix:LDLt Cholesky decomposition? works even when cholesky breaks(LLt)
    Omega.L1 <- ldl(Omega.Psi[1,,])$lower
    Omega.d1 <- ldl(Omega.Psi[1,,])$d
    Omega.Psi[1,,] <- diag(Omega.d1)
    mu.m[1,] <- solve(Omega.L1) %*% apply(data.PCX.transformed[data.clust == 1,], 2, mean)
    mu.C[1,,] <- Omega.Psi[1,,]
    if(vague.mu){mu.C[1,,] <- mu.C[1,,] * model.attributes$S * model.attributes$times}
    for (g in 2:model.attributes$G){
      if(model.attributes$L > 1){Omega.Psi[g,,] <- diag(model.attributes$L)}
      if(model.attributes$L == 1){Omega.Psi[g,,] <- 1}
      mu.m[g,] <- apply(data.PCX.transformed[data.clust == g,], 2, mean)
      mu.C[g,,] <- Omega.Psi[g,,]
      if(vague.mu){mu.C[g,,] <- mu.C[g,,] * model.attributes$S * model.attributes$times}
    }
  }
  # if diag.Psi is false, Omega.Psi, mu.m and mu.C are updated differently depending on
  # whether no. of factors >1 or =1.if factors > 1, update differently for group 1 vs or groups. why?
  if(!diag.Psi){
    for (g in 1:model.attributes$G){
      if(model.attributes$L > 1){
        if(g == 1){
          Omega.Psi[1,,] <- cov(data.PCX.transformed[data.clust == 1,])
          Omega.L1 <- ldl(Omega.Psi[1,,])$lower
          Omega.d1 <- ldl(Omega.Psi[1,,])$d
          Omega.Psi[1,,] <- diag(Omega.d1) 
          mu.m[1,] <- solve(Omega.L1) %*% apply(data.PCX.transformed[data.clust == 1,], 2, mean)
          mu.C[1,,] <- Omega.Psi[1,,] 
          if(vague.mu){mu.C[1,,] <- mu.C[1,,] * model.attributes$S * model.attributes$times}
        }
        if(g > 1){
          Omega.Psi[g,,] <- cov(data.PCX.transformed[data.clust == g,])
          Omega.Psi[g,,] <- solve(Omega.L1) %*% Omega.Psi[g,,] %*% t(solve(Omega.L1)) 
          mu.m[g,] <- solve(Omega.L1) %*% apply(data.PCX.transformed[data.clust == g,], 2, mean)
          mu.C[g,,] <- Omega.Psi[g,,] 
          if(vague.mu){mu.C[g,,] <- mu.C[g,,] * model.attributes$S * model.attributes$times}
        }
      }
      if(model.attributes$L == 1){
        Omega.Psi[g,,] <- var(data.PCX.transformed[data.clust == g,])
        mu.m[g,] <- mean(data.PCX.transformed[data.clust==g,])
        mu.C[g,,] <- Omega.Psi[g,,]
        if(vague.mu){mu.C[g,,] <- mu.C[g,,] * model.attributes$S * model.attributes$times}
      }
    }
  }
  
  if(model.attributes$G > 1 & model.attributes$L > 1){
    for(g in 2:model.attributes$G){
      if(!isSymmetric(Omega.Psi[g,,])){Omega.Psi[g,,][lower.tri(Omega.Psi[g,,])] = t(Omega.Psi[g,,])[lower.tri(Omega.Psi[g,,])]}
    }
  }

  if(zero.mu){
    mu.m <- matrix(0, model.attributes$G, model.attributes$L)
  }
  
  return(list(omega.psi=Omega.Psi,mu.m=mu.m,mu.C=mu.C, p.exponent = p.exponent, initial.cluster = data.clust))
}

