







use std::ptr;




#[must_use]
pub(crate) struct Partial<T> {

    ptr: *mut T,

    pub(crate) len: usize,
}

impl<T> Partial<T> {









    pub(crate) unsafe fn new(ptr: *mut T) -> Self {
        Self {
            ptr,
            len: 0,
        }
    }

    #[cfg(feature = "rayon")]
    pub(crate) fn stub() -> Self {
        Self { len: 0, ptr: ptr::null_mut() }
    }

    #[cfg(feature = "rayon")]
    pub(crate) fn is_stub(&self) -> bool {
        self.ptr.is_null()
    }


    pub(crate) fn release_ownership(mut self) -> usize {
        let ret = self.len;
        self.len = 0;
        ret
    }

    #[cfg(feature = "rayon")]


    pub(crate) fn try_merge(mut left: Self, right: Self) -> Self {
        if !std::mem::needs_drop::<T>() {
            return left;
        }


        if left.is_stub() {
            right
        } else if left.ptr.wrapping_add(left.len) == right.ptr {
            left.len += right.release_ownership();
            left
        } else {

            debug_assert!(false, "Partial: failure to merge left and right parts");
            left
        }
    }
}

unsafe impl<T> Send for Partial<T> where T: Send { }

impl<T> Drop for Partial<T> {
    fn drop(&mut self) {
        if !self.ptr.is_null() {
            unsafe {
                ptr::drop_in_place(alloc::slice::from_raw_parts_mut(self.ptr, self.len));
            }
        }
    }
}
