#' Obtain annotations from 'audioblast.org' data base
#' 
#' \code{find_annotations} downloads sound file annotations and metadata from 'audioblast.org'.
#' @usage find_annotations(qword, parallel = 1, pb = TRUE, warbler.format = FALSE, 
#' download = FALSE, X = NULL, path = NULL)  
#' @param qword Character vector of length one indicating  the scientific name of the species to search for at audioblast's annotations database. For example, \emph{Phaethornis longirostris}. 
#' @param parallel Numeric. Controls whether parallel computing is applied when downloading mp3 files.
#' It specifies the number of cores to be used. Default is 1 (i.e. no parallel computing).
#' @param pb Logical argument to control progress bar. Default is \code{TRUE}.
#' @param warbler.format 	Logical. If \code{TRUE} columns are renamed using the standard names for a selection table as in 'warbleR', frequency limit columns (high and low frequency) in 'Hz' are converted to 'kHz' (as in warbleR selection tables) and the column names are changed to: 'sound.files', 'selec', 'start', 'end', 'bottom.freq' and 'top.freq'. Default is \code{FALSE}.
#' @param download Logical argument. If \code{FALSE} only the recording file names and
#'   associated metadata are downloaded. If \code{TRUE}, recordings are also downloaded to the working
#'   directory. Default is \code{FALSE}.
#' @param X Data frame generated by \code{\link{find_annotations}}. If supplied the recordings listed in the data frame
#' will be downloaded using the 'recording_url' column information (\code{download} argument is automatically set to \code{TRUE}). This can be used to select
#' the recordings to be downloaded based on their attributes. Default is \code{NULL}.
#' @param path Character string containing the directory path where the sound files will be saved. 
#' If \code{NULL} (default) then the current working directory is used.
#' @return A data frame with the annotation information.  
#' @export
#' @name find_annotations
#' @details This function queries for annotations on acoustic media in the open-access
#' online repository 'audioblast.org'.
#' @seealso \code{\link{quer_xc}} 
#' @examples
#'  \dontrun{
#' ann <- find_annotations(qword = "Thyroptera")
#' }
#' @references {
#' Araya-Salas, M., & Smith-Vidaurre, G. (2017). warbleR: An R package to streamline analysis of animal acoustic signals. Methods in Ecology and Evolution, 8(2), 184-191.
#' }
#' @author Marcelo Araya-Salas (\email{marcelo.araya@@ucr.ac.cr}) 
#last modification on dec-24-2019 (MAS)

find_annotations <- function(qword, parallel = 1, pb = TRUE, warbler.format = FALSE, download = FALSE, X = NULL, path = NULL) {
  
  # set pb options 
  on.exit(pbapply::pboptions(type = .Options$pboptions$type), add = TRUE)
  
  #### set arguments from options
  # get function arguments
  argms <- methods::formalArgs(find_annotations)
  
  # get warbleR options
  opt.argms <- if(!is.null(getOption("warbleR"))) getOption("warbleR") else SILLYNAME <- 0
  
  # remove options not as default in call and not in function arguments
  opt.argms <- opt.argms[!sapply(opt.argms, is.null) & names(opt.argms) %in% argms]
  
  # get arguments set in the call
  call.argms <- as.list(base::match.call())[-1]
  
  # remove arguments in options that are in call
  opt.argms <- opt.argms[!names(opt.argms) %in% names(call.argms)]
  
  # set options left
  if (length(opt.argms) > 0)
    for (q in 1:length(opt.argms))
      assign(names(opt.argms)[q], opt.argms[[q]])
  
  #check path to working directory
  if (is.null(path)) path <- getwd() else 
    if (!dir.exists(path)) 
      stop("'path' provided does not exist") else
        path <- normalizePath(path)
  
  #check internet connection
  a <- try(RCurl::getURL("https://api.audioblast.org"), silent = TRUE)
  if (is(a, "try-error")) stop("No connection to api.audioblast.org (check your internet connection!)") else rm(a)
  
  # If parallel is not numeric
  if (!is.numeric(parallel)) stop("'parallel' must be a numeric vector of length 1") 
  if (any(!(parallel %% 1 == 0),parallel < 1)) stop("'parallel' should be a positive integer")
  
  
  if (is.null(X))
  {
  # set pb options 
  pbapply::pboptions(type = ifelse(pb, "timer", "none"))
  
    #search annotations 
    if (pb)
      write(file = "", x = "Obtaining annotations...")
    
    #replace speaces if species or add % at the end if genus 
  if (!is.null(qword))  {
    if (grepl(" ", qword))
      qword <- gsub(" ", "%20", qword) else
        qword <- paste0(qword, "%")
  }
  
    #initialize search
    q <- rjson::fromJSON(file = paste0("https://api.audioblast.org/annotations/?agent=warbleR&taxon=", qword))
    
    if (length(q) == 0) cat("No annotations were found") else {
    
      # set clusters for windows OS
      if (Sys.info()[1] == "Windows" & parallel > 1)
        cl <- parallel::makePSOCKcluster(getOption("cl.cores", parallel)) else cl <- parallel
      
      
      q <- pbapply::pblapply(q, cl = cl,  function(x){
        
        x[sapply(x, is.null)] <- NA
        
        data.frame(x)
        
      })
      
      # determine all column names in all pages
      cnms <- unique(unlist(lapply(q, names)))    
      
      # add columns that are missing to each selection table
      q2 <- lapply(q, function(X)
      {
        nms <- names(X)
        if (length(nms) != length(cnms))  
          for(i in cnms[!cnms %in% nms]) {
            X <- data.frame(X,  NA, stringsAsFactors = FALSE, check.names = FALSE)
            names(X)[ncol(X)] <- i
          }
        return(X)
      })  

        # save results in a single data frame  
        results <- do.call(rbind, q2)
        
        # convert factors to characters
        indx <- sapply(results, is.factor)
        results[indx] <- lapply(results[indx], as.character)
        
        if (pb)
          write(file = "", x = paste0(nrow(results), " annotations found!"))
    
        # convert to numeric
        results$time_start <- as.numeric(results$time_start)
        results$time_end <- as.numeric(results$time_end)
        results$freq_low <- as.numeric(results$freq_low)
        results$freq_high <- as.numeric(results$freq_high)
        
        # change to warbleR format
        if (warbler.format)
        {
          # time
          results$start <- results$time_start
          results$end <- results$time_end
          
          # selection ID
          results$sound.files <- results$source_id
          results$selec <- results$annotation_id
        
          if (any(duplicated(results[, c("sound.files", "selec")])))
          results$selec <- 1:nrow(results)
          
          # frequency
          results$bottom.freq <- results$freq_low / 1000
          results$top.freq <- results$freq_high / 1000  
        
          results$time_start <- results$time_end <- results$source_id <- results$annotation_id <- results$freq_low <- results$freq_high <- NULL
          
          # sort columns intuitively
          results <- warbleR::sort_colms(results)
          }
        } 
  } else { 
    #stop if X is not a data frame
    if (!is(X, "data.frame")) stop("X is not a data frame")
    
    #stop if the basic columns are not found
        if (!"recording_url" %in% colnames(X))
          stop("'recording_url' column not found in 'X'")
    
    download <- TRUE
    results <- X  
  }
  
  # if download or X was supplied
  if(download)
  {
    
    # function to download files
    annFUN <-  function(Y, x){
      if (!file.exists(Y$sound.files[x]))
        download.file(url = Y$recording_url[x], destfile = file.path(path, Y$sound.files[x]),
                      quiet = TRUE,  mode = "wb", cacheOK = TRUE,
                      extra = getOption("download.file.extra"))
      return (NULL)
    }
    
    # add temporary sound files column
    if (is.null(results$sound.files))  results$sound.files <- results$source_id
    
    # keep only one copy of sound files in data frame
    non.dup.res <- results[!duplicated(results$sound.files), ]
    
    # set clusters for windows OS
    if (pb)
      write(file = "", x = "Downloading files...")
    if (Sys.info()[1] == "Windows" & parallel > 1)
      cl <- parallel::makePSOCKcluster(getOption("cl.cores", parallel)) else cl <- parallel
    
    a1 <- pbapply::pblapply(X = 1:nrow(non.dup.res), cl = cl, FUN = function(x) 
    { 
      annFUN(non.dup.res, x) 
    })     
  }
  
  if (!is.null(X)) return(NULL) else{
    if (!warbler.format) results$sound.files <- NULL
    
    return(results)}
}
