% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/racusum_sim.R
\name{racusum_adoc_sim}
\alias{racusum_adoc_sim}
\title{Compute steady-state ARLs of RA-CUSUM control charts using
simulation}
\usage{
racusum_adoc_sim(r, coeff, coeff2, h, df, R0 = 1, RA = 2, RQ = 1,
  m = 50, type = "cond")
}
\arguments{
\item{r}{Integer Vector. Number of runs.}

\item{coeff}{Numeric Vector. Estimated coefficients \eqn{\alpha}{alpha} and \eqn{\beta}{beta}
from the binary logistic regression model.}

\item{coeff2}{Numeric Vector. Estimated coefficients \eqn{\alpha}{alpha} and \eqn{\beta}{beta}
from the binary logistic regression model of a resampled dataset.}

\item{h}{Double. Control Chart limit for detecting deterioration/improvement.}

\item{df}{Data Frame. First column are Parsonnet Score values within a range of \code{0} to
\code{100} representing the preoperative patient risk. The second column are binary (\code{0/1})
 outcome values of each operation.}

\item{R0}{Double. Odds ratio of death under the null hypotheses.}

\item{RA}{Double. Odds ratio of death under the alternative hypotheses. Detecting deterioration
in performance with increased mortality risk by doubling the odds Ratio \code{RA = 2}. Detecting
 improvement in performance with decreased mortality risk by halving the odds ratio of death
 \code{RA = 1/2}.}

\item{RQ}{Double. Defines the performance of a surgeon with the odds ratio ratio of death \code{Q}.}

\item{m}{Integer. Simulated in-control observations.}

\item{type}{Character. Default argument is \code{"cond"} for computation of conditional
steady-state. Other option is the cyclical steady-state \code{"cycl"}.}
}
\value{
Returns a single value which is the Run Length.
}
\description{
Compute steady-state ARLs of risk-adjusted cumulative sum control charts using
 simulation.
}
\examples{
\dontrun{
library(vlad)
data("cardiacsurgery", package="spcadjust")
# build data set
df1 <- subset(cardiacsurgery, select=c(Parsonnet, status))

# estimate coefficients from logit model
coeff1 <- round(coef(glm(status ~ Parsonnet, data=df1, family="binomial")), 3)

# simulation of conditional steady state
m <- 10^3
tau <- 50
res <- sapply(0:(tau-1), function(i){
 RLS <- do.call(c, parallel::mclapply( 1:m, racusum_adoc_sim, RQ=2, h=2.0353, df=df1, m=i,
                                       coeff=coeff1, coeff2=coeff1,
                                       mc.cores=parallel::detectCores()) )
 list(data.frame(cbind(ARL=mean(RLS), ARLSE=sd(RLS)/sqrt(m))))
} )

# plot
RES <- data.frame(cbind(M=0:(tau-1), do.call(rbind, res)))
ggplot2::qplot(x=M, y=ARL, data=RES, geom=c("line", "point")) +
ggplot2::theme_classic()
}
}
\references{
Steiner SH, Cook RJ, Farewell VT and Treasure T (2000).
Monitoring surgical performance using risk-adjusted cumulative sum charts.
\emph{Biostatistics}, \strong{1}(4), pp. 441--452.
doi: \doi{10.1093/biostatistics/1.4.441}.

Wittenberg P, Gan FF, Knoth S (2018).
A simple signaling rule for variable life-adjusted display derived from
an equivalent risk-adjusted CUSUM chart.
\emph{Statistics in Medicine}, \strong{37}(16), pp 2455--2473.

Taylor HM (1968). The Economic Design of Cumulative Sum Control Charts.
\emph{Technometrics}, \strong{10}(3), pp. 479--488.

Crosier R (1986). A new two-sided cumulative quality control scheme.
\emph{Technometrics}, \strong{28}(3), pp. 187--194.
}
\author{
Philipp Wittenberg
}
