% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VCCP.R
\name{vccp.fun}
\alias{vccp.fun}
\title{Multiple change point detection in the vine copula structure of multivariate time series}
\usage{
vccp.fun(
  X,
  method = "NBS",
  delta = 30,
  G = 0.1,
  M = NA,
  test = "V",
  CDR = "D",
  trunc_tree = NA,
  family_set = 1,
  pre_white = 0,
  ar_num = 1,
  p = 0.3,
  N = 100,
  sig_alpha = 0.05
)
}
\arguments{
\item{X}{A numerical matrix representing the multivariate
time series, with the columns representing its components.
If multiple subjects are included (panel data), vertically
stack the subject data and identify timestamps of each subject in the first column.}

\item{method}{A character string, which defines the
segmentation method. If \code{method} = "NBS", which is the
default method, then the adapted new binary segmentation is used.
Similarly, if \code{method}="OBS", "MOSUM" or "WBS", then binary
segmentation, MOSUM and wild binary segmentation are used, respectively.}

\item{delta}{A positive integer number with default value equal to 30.
It is used to define the minimum distance acceptable between
change points. In general, \code{delta} >= 5*ncol(X))
is recommended to ensure sufficient data when estimating the
vine copula model.}

\item{G}{A positive real number between 0 and 1 with default value equal to 0.1.
It is used to define the moving sum bandwidth relative to \code{T} in MOSUM when
\code{method} = "MOSUM" is chosen. Alternatively, a positive integer
less than half of the time series length can be set to define the absolute bandwith.}

\item{M}{A positive integer with default value equal to floor(9*log(T)) (T is the length of the time series).
It represents the number of sub-samples in WBS when
\code{method}="WBS" is chosen.}

\item{test}{A character string, which defines the inference
method used. If \code{test} = "V", which is the default method,
the Vuong test is performed. If \code{test} = "B", the
stationary bootstrap is performed.}

\item{CDR}{A character string, which defines the vine structure.
If \code{CDR} = "D", which is the default method,
a D-vine is used. Similarly, if \code{CDR} = "C" or \code{CDR}
= "R", a C-vine or an R-vine is used, respectively.}

\item{trunc_tree}{A positive integer, which defines the level
of truncation for the vine copula. If \code{trunc_tree} = "NA",
which is the default value, the Vine contains \code{dim(X)[2]-2}
levels of trees.}

\item{family_set}{A positive integer, which defines the bivariate copula
family. If \code{familyset} = 1, which is the default value, only the
Gauss copula is selected and VCCP detects change points in
the linear correlation graph. Coding of pair-copula
families is the same as in \code{\link[VineCopula]{BiCop}}.}

\item{pre_white}{A positive integer, which defines whether
the data is pre-whitened. If \code{pre-white} = 0, which is the
default value, no pre-whitening is performed. If
\code{pre_white} =1, an autoregressive time series model
(method: yule-walker) is used to preprocess the raw data.}

\item{ar_num}{A positive integer, which defines the maximum
order of model to fit to preprocess the data (see \code{pre_white}).
If \code{ar_num} = 1, which is the default value, then an AR(1)
model is fit to the data.}

\item{p}{A positive real number between 0 and 1 which is
defined as the block size in the stationary boostrap
method (\code{rgeom(T,p)}) if \code{test} = "B" is chosen.
If \code{p}=0.3, which is the default value, each resampled block
has 1/0.3 time points on average.}

\item{N}{A positive integer, which defines the number
of the stationary bootstrap resamples used. The default value is \code{N=100}.}

\item{sig_alpha}{A positive real number between 0 and 1, which
defines the significance level of the inference test.
The default values is 0.05.}
}
\value{
A list with the following components:

\tabular{ll}{
\code{loc_of_cpts} \tab The locations of the detected change points. \cr
\code{no_of_cpts} \tab The number of detected change points. \cr
\code{test_df} \tab A dataframe containing the test result.  \cr
\code{compute_time} \tab Time (in minutes), to run \code{vccp.fun}. \cr
\code{T} \tab The length of the time series data. \cr
\code{sig_alpha} \tab The significance level for the inference test. \cr
}
}
\description{
This function detects multiple change points in the vine
copula structure of a multivariate time series using
vine copulas, an adapted binary segmentation algorithm, and a
likelihood ratio test for inference. Other segmentation methods
are also available as well as the stationary bootstrap for inference.
}
\details{
The time series \code{X_t} is of dimensionality p and we are
looking for changes in the vine copula structure between
the different time series components \code{X_{t}^{(1)}, X_{t}^{(2)},
 ..., X_{t}^{(p)}}. VCCP uses vine copulas, an adapted binary segmentation
algorithm, and a likelihood ratio test for inference. Other segmentation
methods are also available as well as the stationary bootstrap for inference.
}
\section{Author(s)}{

Xin Xiong, Ivor Cribben (\email{cribben@ualberta.ca})
}

\section{References}{

"Beyond linear dynamic functional connectivity: a vine copula change point model", Xiong and Cribben (2021), preprint.
}

\examples{
\donttest{
## Simulate MVN data with 2 change points
data <- cbind(1:180, mvn.sim.2.cps(180, 8, seed = 101))
T <- 180
## Change point detection using VCCP (it may take several minutes to complete...)
result.NV <- vccp.fun(data, method = "NBS", delta = 30, test = "V")
## Plot the results
getTestPlot(result.NV)
#title("VCCP: NBS + Vuong")

## Change point detection using NBS and stationary bootstrap for inference
result.NB <- vccp.fun(data, method = "NBS", delta = 30, test = "B")
## Plot the results
getTestPlot(result.NB)
title("VCCP: NBS + Stationary Bootstrap")
}
}
\seealso{
\code{\link{getTestPlot}}
}
